"use client";
import React, { useState, useEffect, useRef, useCallback } from "react";
import Select from "react-select";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import { initialData } from "@/data/data";
import Button from "@/components/shared/button/Button";
import ColorPickerComponent from "@/components/shared/colorpicker/ColorPickerComponent";

// Reusable input component
const InputField = ({ label, id, description, value, onChange, type }) => (
  <div className="flex flex-row w-full mb-4 pb-4 border-b">
    <label htmlFor={id} className="w-1/4 text-gray-400 pr-2">
      <p className="font-bold">{label}</p>
      {description && <p className="text-sm text-gray-600">{description}</p>}
    </label>
    <input
      id={id}
      type={type ?? "text"}
      name={id}
      className="w-3/4 border p-2"
      value={value ?? ""}
      onChange={(e) => onChange(id, e.target.value)}
    />
  </div>
);
const SelectField = ({ label, id, options, value, onChange }) => (
  <div className="flex flex-row w-full mb-4 pb-4 border-b">
    <label htmlFor={id} className="w-1/4 text-gray-400 pr-2">
      <p className="font-bold">{label}</p>
    </label>
    <Select
      options={options}
      className="w-3/4"
      menuPosition={"fixed"}
      name={id}
      placeholder={`Válasszon ${label.toLowerCase()}...`}
      backspaceRemovesValue={false}
      value={options.find((x) => x.value === value) || null}
      onChange={(selected) =>
        onChange(
          id,
          Array.isArray(selected)
            ? selected.map((s) => s.value)
            : selected?.value
        )
      }
    />
  </div>
);
// Reusable color picker
const ColorField = ({ label, id, value, onChange }) => (
  <div className="flex flex-row w-full mb-4 pb-4 border-b">
    <label className="w-1/4 text-gray-400 pr-2">
      <p className="font-bold">{label}</p>
    </label>
    <ColorPickerComponent value={value} onChange={(val) => onChange(id, val)} />
  </div>
);

const LayoutSettingsEditor = () => {
  const [currentSettings, setCurrentSettings] = useState(null);
  const [tab, setTab] = useState(1);
  const inputRefs = useRef([]);
  const index = useRef(0);
  const [formState, setFormState] = useState({});

  const handleInputChange = (key, val) => {
    console.log("🔧 Input Change:", key, val);
    setFormState((prev) => ({
      ...prev,
      [key]: val,
    }));
  };
  const defaultNavSettings = {
    Font: "'Inter', sans-serif",
    FontSize: "16px",
    FontWeight: "400",
    LineHeight: "24px",
    LetterSpacing: "-0.0625rem",
    Background: "#ffffff",
    Color: "#111111",
    darkBackground: "#1a1a1a",
    darkColor: "#f2f2f2",
  };
  const [openTags, setOpenTags] = useState({
    h1: true,
    h2: false,
    h3: false,
    h4: false,
    h5: false,
    h6: false,
  });
  const [openNavTabs, setOpenNavTabs] = useState({
    topNavigation: true,
    centerNavigation: false,
    bottomNavigation: false,
    categorySidemenu: false,
    categoryBreadcrumb: false,
    brandSidemenu: false,
    brandBreadcrumb: false,
    buttonSuccess: false,
    buttonInfo: false,
    buttonCart: false,
    buttonToCart: false,
    buttonToCashier: false,
    buttonRegister: false,
    buttonLogin: false,
  });

  useEffect(() => {
    const loadSettings = async () => {
      const res = await ApiCall.layoutSettings.getLayoutSettings();
      if (res.status === 200 && res.data.settings?.length) {
        const settings = res.data.settings[0];
        setFormState(settings);
        const navTags = [
          "topNavigation",
          "centerNavigation",
          "bottomNavigation",
          "categorySidemenu",
          "categoryBreadcrumb",
          "brandSidemenu",
          "brandBreadcrumb",
          "buttonSuccess",
          "buttonInfo",
          "buttonCart",
          "buttonToCart",
          "buttonToCashier",
          "buttonRegister",
          "buttonLogin",
          "buttonHoverSuccess",
          "buttonHoverInfo",
          "buttonHoverCart",
          "buttonHoverToCart",
          "buttonHoverToCashier",
          "buttonHoverRegister",
          "buttonHoverLogin",
        ];

        setCurrentSettings(settings);
        toast.success("Jelenlegi beállítások sikeresen betöltve.");
      } else {
        toast.error("Sikertelen betöltés!");
      }
    };
    loadSettings();
  }, []);

  const submitLayoutSettings = async () => {
    const content = {
      ...formState,
      type: "layoutSettings",
    };

    console.log("Submitting layout settings", content);
    const res = await ApiCall.layoutSettings.postLayoutSettings(content);
    res.status === 200
      ? toast.success("Sikeres mentés")
      : toast.error("Sikertelen mentés");
  };

  if (!currentSettings)
    return <div className="p-6">Beállítások betöltése...</div>;

  const renderTabs = [
    { id: 1, label: "Layout beállítások" },
    { id: 2, label: "Fejléc betűtípusok" },
    { id: 3, label: "Szöveg beállítások" },
    { id: 4, label: "Link beállítások" },
    { id: 5, label: "Navigáció beállítások" },
    { id: 6, label: "Kategória layout" },
    { id: 7, label: "Brands layout" },
  ];

  const renderLayoutTab = () => (
    <>
      <SelectField
        label="Layout típus kiválasztása"
        id="layoutType"
        options={initialData.settings.layout}
        selected={currentSettings.layoutType}
        value={formState.layoutType}
        onChange={handleInputChange}
      />
      <InputField
        label="Layout szélessége (boxed)"
        id="layoutWidth"
        value={formState.layoutWidth}
        onChange={handleInputChange}
      />
      <SelectField
        label="Tartalom típus kiválasztása"
        id="layoutBodyType"
        options={initialData.settings.layout}
        value={formState.layoutBodyType}
        onChange={handleInputChange}
      />
      <InputField
        label="Tartalom szélessége (boxed)"
        id="layoutBodyWidth"
        value={formState.layoutBodyWidth}
        onChange={handleInputChange}
      />
      <SelectField
        label="Alapértelmezett betűtípus"
        id="fontFamily"
        options={initialData.settings.fontFamily}
        value={formState.fontFamily}
        onChange={handleInputChange}
      />
      <div className="w-full grid grid-cols-2">
        <ColorField
          id={"pageBackground"}
          label="Oldal háttérszín"
          value={formState.pageBackground || "#f2f2f2"}
          onChange={handleInputChange}
        />
        <ColorField
          id={"darkPageBackground"}
          label="Oldal háttérszín - sötét módban"
          value={formState.darkPageBackground || "#212121"}
          onChange={handleInputChange}
        />
      </div>
      <div className="w-full grid grid-cols-2">
        <ColorField
          id={"contentBackground"}
          label="Tartalom háttérszín"
          value={formState.contentBackground}
          onChange={handleInputChange}
        />
        <ColorField
          id={"darkContentBackground"}
          label="Tartalom háttérszín - sötét módban"
          value={formState.darkContentBackground || "#3a3a3a"}
          onChange={handleInputChange}
        />
      </div>
      <div className="w-full grid grid-cols-2">
        <ColorField
          id={"customPageBackground"}
          label="Egyedi oldal háttérszín"
          value={formState.customPageBackground}
          onChange={handleInputChange}
        />
        <ColorField
          id={"darkCustomPageBackground"}
          label="Egyedi oldal háttérszín - sötét módban"
          value={formState.darkCustomPageBackground || "#3a3a3a"}
          onChange={handleInputChange}
        />
      </div>
      <div className="w-full grid grid-cols-2">
        <ColorField
          id={"productCardBackground"}
          label="Termék háttérszín"
          value={formState.productCardBackground}
          onChange={handleInputChange}
        />
        <ColorField
          id={"darkProductCardBackground"}
          label="Termék háttérszín - sötét módban"
          value={formState.darkProductCardBackground || "#3B3B3B"}
          onChange={handleInputChange}
        />
      </div>
    </>
  );
  const renderHeaderFontsTab = () => {
    const toggleTag = (tag) => {
      setOpenTags((prev) => ({ ...prev, [tag]: !prev[tag] }));
    };

    return ["h1", "h2", "h3", "h4", "h5", "h6"].map((tag) => (
      <div key={tag} className="mb-4 border border-gray-200 rounded">
        <button
          type="button"
          onClick={() => toggleTag(tag)}
          className="w-full text-left px-4 py-2 bg-gray-100 font-semibold hover:bg-gray-200 transition">
          {`${tag.toUpperCase()} beállítások `}
          <span className="float-right">{openTags[tag] ? "−" : "+"}</span>
        </button>
        <>{`dark${tag}Color`}</>
        {openTags[tag] && (
          <div className="p-4">
            <SelectField
              label={`${tag.toUpperCase()} betűtípus`}
              id={`${tag}Font`}
              options={initialData.settings.fontFamily}
              value={formState[`${tag}Font`]}
              onChange={handleInputChange}
              // selected={currentSettings[`${tag}Font`]}
              // index={index}
            />
            <InputField
              label={`${tag.toUpperCase()} betűméret`}
              id={`${tag}FontSize`}
              value={formState[`${tag}FontSize`] || "32px"}
              onChange={handleInputChange}
            />
            <SelectField
              label={`${tag.toUpperCase()} betű vastagság`}
              id={`${tag}FontWeight`}
              options={initialData.settings.fontWeight}
              value={formState[`${tag}FontWeight`]}
              onChange={handleInputChange}
            />
            <div className="w-full grid grid-cols-2">
              <ColorField
                id={`${tag}Color`}
                label={`${tag.toUpperCase()} szín`}
                value={formState[`${tag}Color`]}
                onChange={handleInputChange}
              />
              <ColorField
                id={`dark${tag}Color`}
                label={`${tag.toUpperCase()} szín - sötét módban`}
                value={formState[`dark${tag}Color`] ?? "#f2f2f2"}
                onChange={handleInputChange}
              />
            </div>
            <InputField
              label={`${tag.toUpperCase()} sorköz`}
              id={`${tag}LineHeight`}
              value={formState[`${tag}LineHeight`] || "34px"}
              onChange={handleInputChange}
            />
            <InputField
              label={`${tag.toUpperCase()} betűköz`}
              id={`${tag}LetterSpacing`}
              value={formState[`${tag}LetterSpacing`] || "-0.0625rem"}
              onChange={handleInputChange}
            />
          </div>
        )}
      </div>
    ));
  };
  const renderParagraphTab = () => (
    <>
      <SelectField
        label="Szöveg betűtípus"
        id="pFont"
        options={initialData.settings.fontFamily}
        value={formState.pFont}
        onChange={handleInputChange}
      />
      <InputField
        label="Szöveg betűméret"
        id="pFontSize"
        value={formState.pFontSize || "16px"}
        onChange={handleInputChange}
      />
      <InputField
        label={`Kenyérszöveg sorköz`}
        id={`pLineHeight`}
        value={formState[`pLineHeight`] || "24px"}
        onChange={handleInputChange}
      />
      <div className="w-full grid grid-cols-2">
        <ColorField
          id={"pColor"}
          label="Szöveg szín"
          value={formState.pColor}
          onChange={handleInputChange}
        />
        <ColorField
          id={"darkpColor"}
          label="Szöveg szín - sötét módban"
          value={formState.darkpColor || "#f2f2f2"}
          onChange={handleInputChange}
        />
      </div>
    </>
  );
  const renderLinkTab = () => (
    <>
      <SelectField
        label="Link betűtípus"
        id="linkFont"
        options={initialData.settings.fontFamily}
        value={formState.linkFont}
        onChange={handleInputChange}
      />
      <InputField
        label="Link betűméret"
        id="linkFontSize"
        value={formState.linkFontSize || "16px"}
        onChange={handleInputChange}
      />
      <InputField
        label={`Link sorköz`}
        id={`linkLineHeight`}
        value={formState[`linkLineHeight`] || "24px"}
        onChange={handleInputChange}
      />
      <div className="w-full grid grid-cols-2">
        <ColorField
          id={"linkColor"}
          label="Link szín"
          value={formState.linkColor}
          onChange={handleInputChange}
        />
        <ColorField
          id={"darkLinkColor"}
          label="Link szín - sötét módban"
          value={formState.darkLinkColor || "#f2f2f2"}
          onChange={handleInputChange}
        />
      </div>
    </>
  );
  const renderNavigationTab = () => {
    const toggleTag = (tag) => {
      setOpenNavTabs((prev) => ({ ...prev, [tag]: !prev[tag] }));
    };

    return [
      "topNavigation",
      "centerNavigation",
      "bottomNavigation",
      "categorySidemenu",
      "categoryBreadcrumb",
      "brandSidemenu",
      "brandBreadcrumb",
      "buttonSuccess",
      "buttonInfo",
      "buttonCart",
      "buttonToCart",
      "buttonToCashier",
      "buttonRegister",
      "buttonLogin",
      "buttonHoverSuccess",
      "buttonHoverInfo",
      "buttonHoverCart",
      "buttonHoverToCart",
      "buttonHoverToCashier",
      "buttonHoverRegister",
      "buttonHoverLogin",
    ].map((tag) => {
      return (
        <div key={tag} className="mb-4 border border-gray-200 rounded">
          <button
            type="button"
            onClick={() => toggleTag(tag)}
            className="w-full text-left px-4 py-2 bg-gray-100 font-semibold hover:bg-gray-200 transition">
            {`${tag.toUpperCase()} beállítások `}
            <span className="float-right">{openNavTabs[tag] ? "−" : "+"}</span>
          </button>
          {openNavTabs[tag] && (
            <div className="p-4">
              <SelectField
                label={`${tag.toUpperCase()} betűtípus`}
                id={`${tag}Font`}
                options={
                  initialData.settings.fontFamily || [defaultNavSettings.Font]
                }
                value={formState[`${tag}Font`] || defaultNavSettings.Font}
                onChange={handleInputChange}
              />
              <InputField
                label={`${tag.toUpperCase()} betűméret`}
                id={`${tag}FontSize`}
                value={
                  formState[`${tag}FontSize`] || defaultNavSettings.FontSize
                }
                onChange={handleInputChange}
              />
              <SelectField
                label={`${tag.toUpperCase()} betű vastagság`}
                id={`${tag}FontWeight`}
                options={
                  initialData.settings.fontWeight || [
                    defaultNavSettings.FontWeight,
                  ]
                }
                value={
                  formState[`${tag}FontWeight`] || defaultNavSettings.FontWeight
                }
                onChange={handleInputChange}
              />
              <InputField
                label={`${tag.toUpperCase()} sorköz`}
                id={`${tag}LineHeight`}
                value={
                  formState[`${tag}LineHeight`] || defaultNavSettings.LineHeight
                }
                onChange={handleInputChange}
              />
              <InputField
                label={`${tag.toUpperCase()} betűköz`}
                id={`${tag}LetterSpacing`}
                value={
                  formState[`${tag}LetterSpacing`] ||
                  defaultNavSettings.LetterSpacing
                }
                onChange={handleInputChange}
              />
              <div className="w-full grid grid-cols-2">
                <ColorField
                  id={`${tag}Background`}
                  label={`${tag.toUpperCase()} háttér szín`}
                  value={
                    formState[`${tag}Background`] ||
                    defaultNavSettings.Background
                  }
                  onChange={handleInputChange}
                />
                <ColorField
                  id={`dark${tag}Background`}
                  label={`${tag.toUpperCase()} háttér szín - sötét módban`}
                  value={
                    formState[`dark${tag}Background`] ||
                    defaultNavSettings.darkBackground
                  }
                  onChange={handleInputChange}
                />
              </div>
              <div className="w-full grid grid-cols-2">
                <ColorField
                  id={`${tag}Color`}
                  label={`${tag.toUpperCase()} betű szín`}
                  value={formState[`${tag}Color`] || defaultNavSettings.Color}
                  onChange={handleInputChange}
                />
                <ColorField
                  id={`dark${tag}Color`}
                  label={`${tag.toUpperCase()} betű szín - sötét módban`}
                  value={
                    formState[`dark${tag}Color`] || defaultNavSettings.darkColor
                  }
                  onChange={handleInputChange}
                />
              </div>
            </div>
          )}
        </div>
      );
    });
  };
  const renderCategoryLayoutTab = () => {
    return (
      <>
        <InputField
          label="Egyszerre megjelenített kategória elemek"
          id="displayedCategoryElements"
          value={formState.displayedCategoryElements}
          onChange={handleInputChange}
          type="number"
        />
        <InputField
          label="Alapértelmezett GRID oszlop szám"
          id="defaultCategoryGridCount"
          value={formState.defaultCategoryGridCount}
          onChange={handleInputChange}
        />
        <div className="w-full grid grid-cols-2 gap-2">
          <InputField
            label="GRID oszlop szám (XXS)"
            id="categoryGridCountXxs"
            value={formState.categoryGridCountXxs}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID oszlop szám (XS)"
            id="categoryGridCountXs"
            value={formState.categoryGridCountXs}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID oszlop szám (SM)"
            id="categoryGridCountSm"
            value={formState.categoryGridCountSm}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID oszlop szám (MD)"
            id="categoryGridCountMd"
            value={formState.categoryGridCountMd}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID oszlop szám (LG)"
            id="categoryGridCountLg"
            value={formState.categoryGridCountLg}
            onChange={handleInputChange}
          />
        </div>
        <InputField
          label="Alapértelmezett GRID col span érték"
          id="defaultCategoryColSpan"
          value={formState.defaultCategoryColSpan}
          onChange={handleInputChange}
        />
        <div className="w-full grid grid-cols-2 gap-2">
          <InputField
            label="GRID col span érték (XXS)"
            id="categoryColSpanXxs"
            value={formState.categoryColSpanXxs}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID col span érték (XS)"
            id="categoryColSpanXs"
            value={formState.categoryColSpanXs}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID col span érték (SM)"
            id="categoryColSpanSm"
            value={formState.categoryColSpanSm}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID col span érték (MD)"
            id="categoryColSpanMd"
            value={formState.categoryColSpanMd}
            onChange={handleInputChange}
          />
          <InputField
            label="GRID col span érték (LG)"
            id="categoryColSpanLg"
            value={formState.categoryColSpanLg}
            onChange={handleInputChange}
          />
        </div>
      </>
    );
  };
  const renderBrandLayoutTab = () => {
    return <></>;
  };

  return (
    <>
      <div className="column-options-wrapper flex flex-col w-full p-4">
        <div className="flex border-b mb-4">
          {renderTabs.map((tabItem) => (
            <button
              key={tabItem.id}
              onClick={() => setTab(tabItem.id)}
              className={`px-4 py-2 border-t border-l ${
                tab === tabItem.id ? "bg-green-500 text-white" : ""
              }`}>
              {tabItem.label}
            </button>
          ))}
        </div>
        <div className="w-full">
          <div className={tab !== 1 ? "hidden" : ""}>{renderLayoutTab()}</div>
          <div className={tab !== 2 ? "hidden" : ""}>
            {renderHeaderFontsTab()}
          </div>
          <div className={tab !== 3 ? "hidden" : ""}>
            {renderParagraphTab()}
          </div>
          <div className={tab !== 4 ? "hidden" : ""}>{renderLinkTab()}</div>
          <div className={tab !== 5 ? "hidden" : ""}>
            {renderNavigationTab()}
          </div>
          <div className={tab !== 6 ? "hidden" : ""}>
            {renderCategoryLayoutTab()}
          </div>
          <div className={tab !== 7 ? "hidden" : ""}>
            {renderBrandLayoutTab()}
          </div>
        </div>
      </div>
      <div className="flex justify-end sticky bottom-0 bg-white p-2 border-t">
        <Button
          onClick={submitLayoutSettings}
          className="px-4 py-2 uppercase border bg-green-500/75 text-white hover:border-green-500 hover:bg-green-500"
          btntext="Mentés"
        />
      </div>
    </>
  );
};

export default LayoutSettingsEditor;
