// src/app/admin/hr/doc-templates/page.tsx
"use client";

import { useEffect, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import { TemplateEditor } from "@/components/shared/texteditor/TemplateEditor";

type HrTemplateRow = {
  id: string;
  title: string;
  is_active: boolean;
  updated_at: string | null;
};

export default function HrTemplateDocumentsPage() {
  const [rows, setRows] = useState<HrTemplateRow[]>([]);
  const [loading, setLoading] = useState(false);
  const [editingId, setEditingId] = useState<string | null>(null);

  const loadAll = async () => {
    setLoading(true);
    try {
      // BE: listTemplateDocuments with kind_code=hr_person_doc, is_active=all
      const res = await ApiCall.templateDocuments.getList({
        kind_code: "hr_person_doc",
        is_active: "all",
      });

      // backend returns plain array
      const data: HrTemplateRow[] = res.data ?? [];
      setRows(data);
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.error ||
          "HR dokumentumsablonok betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadAll();
  }, []);

  const handleDelete = async (row: HrTemplateRow) => {
    if (!confirm(`Biztosan törlöd a sablont? (${row.title})`)) return;
    try {
      await ApiCall.templateDocuments.removeOne(row.id);
      toast.success("Dokumentumsablon törölve");
      if (editingId === row.id) setEditingId(null);
      await loadAll();
    } catch (err: any) {
      console.error(err);
      toast.error(
        err?.response?.data?.error || "Dokumentumsablon törlése sikertelen"
      );
    }
  };

  const columns = [
    {
      id: "title",
      name: "Megnevezés",
      selector: (r: HrTemplateRow) => r.title,
      sortable: true,
      grow: 2,
    },
    {
      id: "updated_at",
      name: "Utoljára módosítva",
      selector: (r: HrTemplateRow) => r.updated_at || "",
      cell: (r: HrTemplateRow) =>
        r.updated_at ? new Date(r.updated_at).toLocaleString("hu-HU") : "—",
      sortable: true,
      width: "220px",
    },
    {
      id: "is_active",
      name: "Aktív?",
      selector: (r: HrTemplateRow) => (r.is_active ? "Igen" : "Nem"),
      cell: (r: HrTemplateRow) => (r.is_active ? "✅" : "—"),
      width: "80px",
    },
    {
      id: "actions",
      name: "Művelet",
      cell: (r: HrTemplateRow) => (
        <div className="flex gap-2">
          <button
            type="button"
            className="px-2 py-1 text-xs rounded bg-blue-600 text-white"
            onClick={() => setEditingId(r.id)}>
            Szerkesztés
          </button>
          <button
            type="button"
            className="px-2 py-1 text-xs rounded bg-red-600 text-white"
            onClick={() => handleDelete(r)}>
            Törlés
          </button>
        </div>
      ),
      width: "180px",
    },
  ];

  const handleNew = () => setEditingId(null);

  const handleSaved = () => {
    loadAll();
  };

  return (
    <main className="p-4 space-y-4">
      <h1 className="text-lg font-semibold">HR – Dokumentum sablonok</h1>

      {/* List */}
      <DataTableWrapper
        tableTitle="Dokumentum sablonok (TemplateDocuments)"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        pagination
        keyField="id"
      />

      {/* Editor */}
      <section className="mt-4 border rounded p-4 bg-gray-50 space-y-3">
        <div className="flex justify-between items-center">
          <h2 className="font-semibold text-sm">
            {editingId
              ? "Dokumentumsablon szerkesztése"
              : "Új dokumentumsablon létrehozása"}
          </h2>
          <button
            type="button"
            className="text-xs text-gray-500 underline"
            onClick={handleNew}>
            Új sablon mód
          </button>
        </div>

        <TemplateEditor
          id={editingId ?? undefined}
          // templateKind is used for header/footer 'doc_templates' filtering,
          // you can pass whatever kind you use there (or leave undefined).
          templateKind="hr_person_doc" // or "company_header_footer" or whatever you use
          onSaved={handleSaved}
        />
      </section>
    </main>
  );
}
