"use client";

import { useEffect, useMemo, useState, useCallback } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import toast from "react-hot-toast";
import useTableDebounce from "@/hooks/useTableDebounce";

const FrontendMessagesList = () => {
  const token = ""; // Optional: auth token
  const [data, setData] = useState([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("messageType");
  const [sortDirection, setSortDirection] = useState("asc");
  const [totalRecords, setTotalRecords] = useState(0);
  const [inputValue, setInputValue] = useState("");
  const [selectedRows, setSelectedRows] = useState([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const debouncedSearchTerm = useTableDebounce(inputValue, 500); // ✅ Use custom hook

  const query = `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;

  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await ApiCall.messages.getFrontendMessagesList(query);
      setData(response?.data?.tableData ?? []);
      setTotalRecords(response?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
  }, [query]);

  const debouncedSearch = useCallback((value) => {
    const delay = setTimeout(() => {
      setSearchTerm(value);
      setPage(1);
    }, 500);
    return () => clearTimeout(delay);
  }, []);

  const handleSelectAction = async (actionType, rowsOrRow) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r._id);

    if (actionType === "delete") {
      const confirmDelete = confirm(
        `Biztosan törölni szeretnéd a következő elemeket?\n${ids.join(", ")}`
      );
      if (confirmDelete) {
        try {
          const response = await ApiCall.messages.delete(ids);
          if (response.status === 200) {
            toast.success(response?.data?.message);
          } else {
            toast.error(response?.data?.error);
          }
          fetchData();
          setToggleCleared(!toggleCleared);
        } catch (err) {
          console.error("Delete failed:", err);
        }
      }
    }

    if (actionType === "edit") {
      if (selected.length !== 1) {
        alert("Csak egy elem szerkeszthető egyszerre.");
        return;
      }
      const id = selected[0]._id;
      window.location.href = `/admin/frontend-messages/edit/${id}`;
    }
  };

  const tableSelectOptions = [
    {
      text: "Szerkesztés",
      type: "edit",
      className: "bg-blue-500 text-white",
    },
    {
      text: "Törlés",
      type: "delete",
      className: "bg-red-500 text-white",
    },
  ];

  const columns = useMemo(
    () => [
      {
        id: "_id",
        name: "Üzenet id",
        selector: (row) => row._id,
        sortable: true,
      },
      {
        id: "messageType",
        name: "Üzenet típusa",
        selector: (row) => row.messageType,
        sortable: true,
      },
      {
        id: "messageActive",
        name: "Üzenet aktív az oldalon",
        selector: (row) => {
          if (row.messageActive) {
            return <p className="text-green-600">Aktív</p>;
          } else {
            return <p className="text-red-600">Inaktív</p>;
          }
        },
        sortable: true,
      },
      {
        id: "createdAt",
        name: "Létrehozva",
        selector: (row) => row.createdAt,
        sortable: true,
      },
      {
        id: "updatedAt",
        name: "Frissítve",
        selector: (row) => row.updatedAt,
        sortable: true,
      },
    ],
    []
  );

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => {
          setInputValue(newInputValue);
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
      {selectedRows.length > 0 && (
        <div className="flex items-center gap-2 bg-gray-100 px-4 py-2 border rounded">
          <span className="text-sm text-gray-600">
            {selectedRows.length} kiválasztva
          </span>
          {tableSelectOptions.map((option) => (
            <button
              key={option.text}
              onClick={() => {
                handleSelectAction(option.type, selectedRows);
              }}
              className={`rounded px-3 py-1 text-sm ${option.className}`}>
              {option.text}
            </button>
          ))}
        </div>
      )}
    </div>
  );

  return (
    <DataTableWrapper
      tableTitle="Weboldal üzenetek"
      data={data}
      columns={columns}
      progressPending={loading}
      paginationDefaultPage={page}
      paginationServer
      onChangePage={(pageNumber) => setPage(pageNumber)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col, dir) => {
        setSortField(col.id);
        setSortDirection(dir);
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      tableOptionsPath="frontend-messages/create"
      selectableRows={true}
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
    />
  );
};

export default FrontendMessagesList;
