// ==============================================
// File: app/admin/email-registry/list/page.tsx
// ==============================================
"use client";

import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import toast from "react-hot-toast";
import Select from "react-select";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import ExpandedEmailRegistry from "@/components/email-registry/ExpandedEmailRegistry";
import { useRouter } from "next/navigation";

const huStatus = (s?: Row["user_status"] | string | null) => {
  const v = String(s ?? "")
    .trim()
    .toLowerCase();
  if (["active", "aktív", "aktiv"].includes(v)) return "Aktív";
  if (
    ["quit", "kilépett", "kilepett", "inactive", "inaktív", "inaktiv"].includes(
      v
    )
  )
    return "Kilépett";
  return ""; // unknown / missing
};

// export type Row = {
//   id: string;
//   email: string;
//   type?: string | null;
//   handler_person?: string | null;
//   handler_segment?: string | null;
//   user_status?: "Aktív" | "Kilépett" | "active" | "quit" | null; // 👈 NEW
//   groups: string[];
//   is_forwarded: boolean;
//   forward_to?: string | null;
//   should_archive: boolean;
//   notes?: string | null;
//   updated_at?: string | null;
// };
export interface Row {
  id: string;
  email: string;
  type_id: string | null;
  type: string | null;
  handler_person_id: string | null;
  handler_person: string | null;
  handler_segment_id: string | null;
  handler_segment: string | null;
  user_status_id: string | null;
  user_status: string;
  groups: string[];
  is_forwarded: boolean;
  forward_to: string | null;
  should_archive: boolean;
  notes: string | null;
  receive_status: string | null;
  send_status: string | null;
  login_status: string | null;
  updated_at: string;

  // 🔵 new fields (all optional to avoid breaking existing code)
  cpanel_exists?: boolean;
  cpanel_email?: string | null;
  cpanel_domain?: string | null;
  cpanel_user?: string | null;
  cpanel_diskquota?: number | string | null;
  cpanel_diskused?: number | string | null;
  cpanel_suspended_login?: number | null;
  cpanel_suspended_incoming?: number | null;
  cpanel_suspended_outgoing?: number | null;
  cpanel_status?: "ok" | "missing" | "suspended" | string;
}

export type Option = { value: string; label: string };

type EmailUploadSummary = {
  processed: number;
  inserted: number;
  updated: number;

  typesCreated: number;
  peopleUpserted: number;
  groupsCreated: number;
  groupLinksInserted: number;
  groupLinksCleared: number;

  forwardsActivated: number;
  forwardsInserted: number;
  forwardsDeactivated: number;
  forwardTargetsReplaced: number;

  userStatusCreated: number;

  warnings: string[];
};
export default function EmailRegistryListPage() {
  const router = useRouter();
  const [rows, setRows] = useState<Row[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(25);
  const [total, setTotal] = useState(0);

  // filters
  const [search, setSearch] = useState("");
  const [typeOptions, setTypeOptions] = useState<Option[]>([]);
  const [groupOptions, setGroupOptions] = useState<Option[]>([]);
  const [selectedTypes, setSelectedTypes] = useState<Option[] | null>(null);
  const [selectedGroups, setSelectedGroups] = useState<Option[] | null>(null);
  const [onlyForwarded, setOnlyForwarded] = useState<"all" | "yes" | "no">(
    "all"
  );
  const [onlyArchive, setOnlyArchive] = useState<"all" | "yes" | "no">("all");
  const [onlyStatus, setOnlyStatus] = useState<"all" | "active" | "quit">(
    "all"
  );

  const [sortField, setSortField] = useState<
    "email" | "updated_at" | "type" | "handler" | "user_status"
  >("updated_at");
  const [sortDir, setSortDir] = useState<"asc" | "desc">("desc");

  // expanded row
  const [expandedRow, setExpandedRow] = useState<Row | null>(null);
  const expandedRowRef = useRef<HTMLDivElement | null>(null);
  const fileRef = useRef<HTMLInputElement | null>(null);
  const [emailUploadReport, setEmailUploadReport] =
    useState<EmailUploadSummary | null>(null);
  const emailFileRef = useRef<HTMLInputElement | null>(null);

  const onEmailFileClick = () => {
    if (!emailFileRef.current) return;
    emailFileRef.current.value = "";
    emailFileRef.current.click();
  };

  const copyEmailWarnings = async (warnings: string[]) => {
    try {
      await navigator.clipboard.writeText((warnings || []).join("\n"));
      toast.success("Figyelmeztetések vágólapra másolva");
    } catch {
      toast.error("Másolás nem sikerült");
    }
  };

  const downloadEmailWarningsTxt = (warnings: string[]) => {
    const blob = new Blob([(warnings || []).join("\n")], {
      type: "text/plain;charset=utf-8",
    });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "email_upload_warnings.txt";
    a.click();
    URL.revokeObjectURL(url);
  };
  // fetch filters once
  useEffect(() => {
    (async () => {
      try {
        const [typesRes, groupsRes] = await Promise.all([
          ApiCall.emailRegistry.getTypes(),
          ApiCall.emailRegistry.getGroups(),
        ]);
        // console.log("type", typesRes);
        setTypeOptions(
          (typesRes.data?.items ?? []).map((t: any) => ({
            value: t.value,
            label: t.label,
          }))
        );
        setGroupOptions(
          (groupsRes.data?.items ?? []).map((g: any) => ({
            value: g.id,
            label: g.label,
          }))
        );
      } catch (e) {
        console.error(e);
        toast.error("Szűrők betöltése sikertelen");
      }
    })();
  }, []);

  // build query string
  const queryString = useMemo(() => {
    const p = new URLSearchParams();
    p.set("page", String(page));
    p.set("pageSize", String(pageSize));
    if (search.trim()) p.set("search", search.trim());
    p.set("sortField", sortField);
    p.set("sortDir", sortDir);
    if (selectedTypes?.length)
      p.set("types", selectedTypes.map((t) => t.value).join(","));
    if (selectedGroups?.length)
      p.set("groupIds", selectedGroups.map((g) => g.value).join(","));
    if (onlyForwarded !== "all") p.set("forwarded", onlyForwarded);
    if (onlyArchive !== "all") p.set("archive", onlyArchive);

    // 👇 NEW
    if (onlyStatus !== "all") p.set("status", onlyStatus);

    return `?${p.toString()}`;
  }, [
    page,
    pageSize,
    search,
    selectedTypes,
    selectedGroups,
    onlyForwarded,
    onlyArchive,
    onlyStatus, // 👈 NEW
    sortField,
    sortDir,
  ]);

  // fetch table data
  const fetchData = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.emailRegistry.getList(queryString);
      setRows(res.data?.tableData ?? []);
      setTotal(res.data?.totalRecords ?? 0);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Lista betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, [queryString]);

  // export & upload
  const handleExportCsv = async () => {
    try {
      const res = await ApiCall.emailRegistry.exportCsv(queryString);
      const blob = new Blob([res.data], { type: "text/csv;charset=utf-8" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = "email_registry.csv";
      a.click();
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("CSV export sikertelen");
    }
  };

  const handleExportXlsx = async () => {
    try {
      const res = await ApiCall.emailRegistry.exportXlsx(queryString);
      const blob = new Blob([res.data], {
        type: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
      });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = "email_registry.xlsx";
      a.click();
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("XLSX export sikertelen");
    }
  };

  const onFileClick = () => {
    if (!fileRef.current) return;
    // reset so selecting the same file triggers onChange again
    fileRef.current.value = "";
    fileRef.current.click();
  };
  const onFileChange: React.ChangeEventHandler<HTMLInputElement> = (e) => {
    const f = e.target.files?.[0];
    if (!f) return;
    // optional: console to verify
    // console.log("Selected file:", f.name, f.type, f.size);
    handleEmailUpload(f);
  };
  const handleEmailUpload = async (file: File) => {
    if (!file) return;
    const form = new FormData();
    form.append("file", file);
    try {
      const res = await ApiCall.emailRegistry.upload(form); // <-- implement in your ApiCall layer
      const summary: EmailUploadSummary | undefined = res?.data?.summary;

      if (summary) {
        setEmailUploadReport(summary);
        toast.success(
          `Import kész • Össz: ${summary.processed}, Új: ${summary.inserted}, Frissítve: ${summary.updated}`
        );
      } else {
        setEmailUploadReport(null);
        toast.success("Import kész");
      }

      // refresh your table/list here
      fetchData?.();
    } catch (e: any) {
      console.error(e);
      const msg =
        e?.response?.data?.error ||
        e?.response?.data?.details ||
        "E-mail import sikertelen";
      toast.error(msg);
    }
  };

  const onEmailFileChange: React.ChangeEventHandler<HTMLInputElement> = (e) => {
    const f = e.target.files?.[0];
    if (!f) return;
    handleEmailUpload(f);
  };

  // columns
  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  const formatCpanelDisk = (r: Row) => {
    if (!r.cpanel_exists || r.cpanel_diskquota == null) return "–";

    // Handle string or number from backend
    const rawUsed = r.cpanel_diskused as any;
    const rawQuota = r.cpanel_diskquota as any;

    const used = rawUsed == null ? 0 : Number(rawUsed);
    const quota = rawQuota == null ? null : Number(rawQuota);

    if (!isFinite(used) || used < 0) return "–";

    // cPanel returns MB; 0 or null quota = unlimited
    const usedMB = Math.round(used); // already MB
    const quotaMB = quota && quota > 0 ? Math.round(quota) : null;

    if (quotaMB === null) {
      return `${usedMB} MB / ∞`;
    }

    const pct = quotaMB > 0 ? Math.round((usedMB / quotaMB) * 100) : 0;

    return `${usedMB} / ${quotaMB} MB (${pct}%)`;
  };

  const cpanelStatusLabel = (r: Row) => {
    const s = (r.cpanel_status || (r.cpanel_exists ? "ok" : "missing")) as
      | "ok"
      | "missing"
      | "suspended"
      | string;

    if (s === "ok") return "OK";
    if (s === "suspended") return "Felfüggesztve";
    if (s === "missing") return "Nincs cPanel";

    return s;
  };

  const columns = [
    {
      id: "email",
      name: "E-mail",
      selector: (r: Row) => r.email,
      cell: (r: Row) => ellipsis(r.email),
      sortable: true,
      width: "200px", // fixed prevents jitter
      // or minWidth: "280px", maxWidth: "360px",
    },
    {
      id: "type",
      name: "Típus",
      selector: (r: Row) => r.type ?? "",
      cell: (r: Row) => ellipsis(r.type ?? ""),
      sortable: true,
      width: "160px",
    },
    {
      id: "groups",
      name: "Cégcsoport(ok)",
      selector: (r: Row) => r.groups.join(", "),
      cell: (r: Row) => ellipsis(r.groups.join(", ")),
      sortable: true,
      width: "200px",
    },
    {
      id: "handler_segment",
      name: "Kezelő Csoport",
      selector: (r: Row) => r.handler_segment ?? "",
      cell: (r: Row) => ellipsis(r.handler_segment ?? ""),
      sortable: true,
      width: "200px",
    },
    {
      id: "handler_person",
      name: "Kezelő Személy",
      selector: (r: Row) => r.handler_person ?? "",
      cell: (r: Row) => ellipsis(r.handler_person ?? ""),
      sortable: true,
      width: "200px",
    },

    {
      id: "user_status",
      name: "Felhasználói státusz",
      selector: (r: Row) => huStatus(r.user_status),
      cell: (r: Row) => ellipsis(huStatus(r.user_status)),
      sortable: true,
      width: "140px",
    },

    {
      id: "cpanel_status",
      name: "cPanel",
      selector: (r: Row) => cpanelStatusLabel(r),
      cell: (r: Row) => {
        const s = (r.cpanel_status || (r.cpanel_exists ? "ok" : "missing")) as
          | "ok"
          | "missing"
          | "suspended";

        let bg = "#E5E7EB"; // gray
        let fg = "#374151";

        if (s === "ok") {
          bg = "#DCFCE7"; // green-ish
          fg = "#166534";
        } else if (s === "suspended") {
          bg = "#FEE2E2"; // red-ish
          fg = "#991B1B";
        } else if (s === "missing") {
          bg = "#FEF3C7"; // yellow-ish
          fg = "#92400E";
        }

        return (
          <span
            title={cpanelStatusLabel(r)}
            style={{
              display: "inline-flex",
              padding: "0.15rem 0.5rem",
              borderRadius: "999px",
              fontSize: "0.75rem",
              fontWeight: 500,
              backgroundColor: bg,
              color: fg,
              whiteSpace: "nowrap",
            }}>
            {cpanelStatusLabel(r)}
          </span>
        );
      },
      sortable: false,
      width: "140px",
    },
    {
      id: "cpanel_disk",
      name: "Tárhely (cPanel)",
      selector: (r: Row) => formatCpanelDisk(r),
      cell: (r: Row) => (
        <div style={{ textAlign: "right" }}>
          {ellipsis(formatCpanelDisk(r))}
        </div>
      ),
      sortable: false,
      width: "200px",
    },

    {
      id: "forward",
      name: "Átirányítás",
      selector: (r: Row) =>
        r.is_forwarded
          ? r.forward_to
            ? `Igen → ${r.forward_to}`
            : "Igen"
          : "Nem",
      // let this one flex a bit more, but still truncate
      cell: (r: Row) =>
        ellipsis(
          r.is_forwarded
            ? r.forward_to
              ? `Igen → ${r.forward_to}`
              : "Igen"
            : "Nem"
        ),
      sortable: true,
      // minWidth: "220px",
      grow: 2, // this column can use leftover space
      width: "160px",
    },
    {
      id: "archive",
      name: "Archiválni",
      selector: (r: Row) => (r.should_archive ? "Igen" : "Nem"),
      cell: (r: Row) => ellipsis(r.should_archive ? "Igen" : "Nem"),
      sortable: true,
      width: "120px",
    },
    {
      id: "updated_at",
      name: "Módosítva",
      selector: (r: Row) => r.updated_at ?? "",
      cell: (r: Row) => (
        <div style={{ textAlign: "right" }}>{ellipsis(r.updated_at ?? "")}</div>
      ),
      sortable: true,
      width: "160px",
    },
  ];
  // const columns = [
  //   {
  //     id: "email",
  //     name: "E-mail",
  //     selector: (r: Row) => r.email,
  //     sortable: true,
  //   },
  //   {
  //     id: "type",
  //     name: "Típus",
  //     selector: (r: Row) => r.type ?? "",
  //     sortable: true,
  //   },
  //   {
  //     id: "handler",
  //     name: "Kezelő",
  //     selector: (r: Row) => r.handler ?? "",
  //     sortable: true,
  //   },
  //   {
  //     id: "user_status",
  //     name: "Felhasználói státusz",
  //     selector: (r: Row) => huStatus(r.user_status),
  //     sortable: true,
  //   },

  //   {
  //     id: "groups",
  //     name: "Cégcsoport(ok)",
  //     selector: (r: Row) => r.groups.join(", "),
  //     sortable: true,
  //   },
  //   {
  //     id: "forward",
  //     name: "Átirányítás",
  //     selector: (r: Row) =>
  //       r.is_forwarded
  //         ? r.forward_to
  //           ? `Igen → ${r.forward_to}`
  //           : "Igen"
  //         : "Nem",
  //     sortable: true,
  //   },
  //   {
  //     id: "archive",
  //     name: "Archiválni",
  //     selector: (r: Row) => (r.should_archive ? "Igen" : "Nem"),
  //     sortable: true,
  //   },
  //   {
  //     id: "updated_at",
  //     name: "Módosítva",
  //     selector: (r: Row) => r.updated_at ?? "",
  //     sortable: true,
  //   },
  // ];
  const conditionalRowStyles = [
    {
      when: (r: Row) => huStatus(r.user_status) === "Kilépett",
      style: {
        backgroundColor: "#FFF1F2",
        color: "#7F1D1D",
      },
    },
    {
      when: (r: Row) => {
        const s = (r.cpanel_status || (r.cpanel_exists ? "ok" : "missing")) as
          | "ok"
          | "missing"
          | "suspended"
          | string;

        return s === "suspended";
      },
      style: {
        backgroundColor: "#FFF1F2",
        color: "#7F1D1D",
      },
    },
  ];
  // expanded renderer
  const renderExpanded = useCallback(
    (row: Row, onSaved: () => void) => (
      <div ref={expandedRowRef as any} className="p-3">
        <ExpandedEmailRegistry
          id={row.id}
          row={row}
          onSaved={() => {
            onSaved?.();
            fetchData();
          }}
          onDeleted={() => {
            setExpandedRow(null);
            fetchData();
          }}
        />
      </div>
    ),
    []
  );

  return (
    <main className="p-4 space-y-4">
      {/* Controls */}
      <div className="flex flex-wrap gap-2 items-center">
        <input
          className="border rounded px-3 py-2"
          placeholder="Keresés (e-mail, kezelő, megjegyzés)"
          value={search}
          onChange={(e) => {
            setSearch(e.target.value);
            setPage(1);
          }}
        />
        <select
          className="border rounded px-3 py-2"
          value={onlyStatus}
          onChange={(e) => {
            setOnlyStatus(e.target.value as any);
            setPage(1);
          }}>
          <option value="all">Státusz: mind</option>
          <option value="active">Aktív</option>
          <option value="quit">Kilépett</option>
        </select>
        <Select
          className="min-w-48"
          isMulti
          options={typeOptions}
          value={selectedTypes ?? []}
          placeholder="Típus szűrő"
          onChange={(v) => {
            setSelectedTypes(v as Option[]);
            setPage(1);
          }}
        />
        <Select
          className="min-w-48"
          isMulti
          options={groupOptions}
          value={selectedGroups ?? []}
          placeholder="Cégcsoport szűrő"
          onChange={(v) => {
            setSelectedGroups(v as Option[]);
            setPage(1);
          }}
        />
        <select
          className="border rounded px-3 py-2"
          value={onlyForwarded}
          onChange={(e) => {
            setOnlyForwarded(e.target.value as any);
            setPage(1);
          }}>
          <option value="all">Átirányítás: mind</option>
          <option value="yes">Igen</option>
          <option value="no">Nem</option>
        </select>
        <select
          className="border rounded px-3 py-2"
          value={onlyArchive}
          onChange={(e) => {
            setOnlyArchive(e.target.value as any);
            setPage(1);
          }}>
          <option value="all">Archiválni: mind</option>
          <option value="yes">Igen</option>
          <option value="no">Nem</option>
        </select>

        <div className="flex gap-2 ml-auto">
          {/* ➕ New */}
          <button
            onClick={() => router.push("/admin/email-registry/create")}
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded">
            Új e-mail
          </button>
          <button
            onClick={handleExportCsv}
            className="px-3 py-2 bg-blue-500/80 hover:bg-blue-500 text-white rounded">
            CSV export
          </button>
          <button
            onClick={handleExportXlsx}
            className="px-3 py-2 bg-green-600/80 hover:bg-green-600 text-white rounded">
            XLSX export
          </button>
          <button
            type="button"
            onClick={onEmailFileClick}
            className="px-3 py-2 bg-gray-700/80 hover:bg-gray-700 text-white rounded">
            E-mail feltöltés (XLSX)
          </button>
          <input
            ref={emailFileRef}
            type="file"
            accept=".xlsx,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
            onChange={onEmailFileChange}
            className="sr-only"
          />
          {/* <label className="px-3 py-2 bg-gray-700/80 hover:bg-gray-700 text-white rounded cursor-pointer">
            Feltöltés (XLSX)
            <input
              type="file"
              className="hidden"
              accept=".xlsx, application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
              onChange={
                (e) => console.log("!!!!!!!!!!!!!!!!!!!!!!!!!", e.target.files)
                // e.target.files && handleUpload(e.target.files[0])
              }
            />
          </label> */}
        </div>
      </div>
      {emailUploadReport && (
        <div className="rounded-md border border-amber-300 bg-amber-50 p-3">
          <div className="flex items-start gap-3">
            <div className="flex-1">
              <div className="font-medium text-amber-900">
                E-mail feltöltés összegzés
              </div>

              <div className="mt-1 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-x-6 gap-y-1 text-sm text-amber-900/90">
                <div>
                  Feldolgozva: <b>{emailUploadReport.processed}</b>
                </div>
                <div>
                  Új: <b>{emailUploadReport.inserted}</b>
                </div>
                <div>
                  Frissítve: <b>{emailUploadReport.updated}</b>
                </div>

                <div>
                  Típus létrehozva: <b>{emailUploadReport.typesCreated}</b>
                </div>
                <div>
                  Személy rögzítve: <b>{emailUploadReport.peopleUpserted}</b>
                </div>
                <div>
                  Cégcsoport létrehozva:{" "}
                  <b>{emailUploadReport.groupsCreated}</b>
                </div>
                <div>
                  Csoport link beszúrva:{" "}
                  <b>{emailUploadReport.groupLinksInserted}</b>
                </div>
                <div>
                  Csoport link törölve:{" "}
                  <b>{emailUploadReport.groupLinksCleared}</b>
                </div>

                <div>
                  Átirányítás aktiválva:{" "}
                  <b>{emailUploadReport.forwardsActivated}</b>
                </div>
                <div>
                  Átirányítás újonnan:{" "}
                  <b>{emailUploadReport.forwardsInserted}</b>
                </div>
                <div>
                  Átirányítás kikapcsolva:{" "}
                  <b>{emailUploadReport.forwardsDeactivated}</b>
                </div>
                <div>
                  Célcímek cserélve:{" "}
                  <b>{emailUploadReport.forwardTargetsReplaced}</b>
                </div>

                <div>
                  Felhasználói státusz létrehozva:{" "}
                  <b>{emailUploadReport.userStatusCreated}</b>
                </div>
                <div>
                  Figyelmeztetések:{" "}
                  <b>{emailUploadReport.warnings?.length || 0}</b>
                </div>
              </div>

              {emailUploadReport.warnings?.length > 0 && (
                <details className="mt-2 group">
                  <summary className="cursor-pointer select-none text-sm text-amber-900 underline underline-offset-2">
                    Figyelmeztetések megnyitása
                  </summary>
                  <div className="mt-2 max-h-64 overflow-auto rounded border bg-white/70 p-2 text-sm">
                    <ul className="list-disc pl-5 space-y-1">
                      {emailUploadReport.warnings.map((w, i) => (
                        <li key={i} className="text-gray-800">
                          {w}
                        </li>
                      ))}
                    </ul>
                  </div>
                </details>
              )}
            </div>

            <div className="flex shrink-0 flex-col gap-2">
              <button
                onClick={() =>
                  copyEmailWarnings(emailUploadReport.warnings || [])
                }
                disabled={!emailUploadReport.warnings?.length}
                className="px-3 py-1.5 rounded bg-gray-700 text-white disabled:opacity-40">
                Másolás
              </button>
              <button
                onClick={() =>
                  downloadEmailWarningsTxt(emailUploadReport.warnings || [])
                }
                disabled={!emailUploadReport.warnings?.length}
                className="px-3 py-1.5 rounded bg-gray-700 text-white disabled:opacity-40">
                Letöltés (TXT)
              </button>
              <button
                onClick={() => setEmailUploadReport(null)}
                className="px-3 py-1.5 rounded border border-gray-300">
                Bezár
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Table */}
      <DataTableWrapper
        tableTitle="E-mail nyilvántartás"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        paginationServer
        paginationDefaultPage={page}
        paginationPerPage={pageSize}
        paginationTotalRows={total}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        sortServer
        defaultSortFieldId={sortField}
        onSort={(col: any, dir: "asc" | "desc") => {
          setSortField((col?.id as any) ?? "email");
          setSortDir(dir);
          setPage(1);
        }}
        expandableRows
        expandableRowExpanded={(r: Row) => r?.id === expandedRow?.id}
        onRowExpandToggled={(open: boolean, row: Row) => {
          if (open) {
            if (expandedRow && expandedRow.id !== row.id) setExpandedRow(null);
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: Row) =>
          renderExpanded(row, () => {
            toast.success("Mentve");
            fetchData();
          })
        }
        conditionalRowStyles={conditionalRowStyles}
      />
    </main>
  );
}
