"use client";

import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import { useRouter } from "next/navigation";
import dynamic from "next/dynamic";
import toast from "react-hot-toast";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import AsyncSelect from "react-select/async";

const ExpandedDomainRegistry = dynamic(
  () => import("@/components/domain-registry/ExpandedDomainRegistry"),
  {
    ssr: false,
    loading: () => <div className="p-3 text-sm text-gray-500">Betöltés…</div>,
  }
);

export type Row = {
  id: string;
  domain_name: string;
  current_company_id?: string | null;
  current_company_name?: string | null;
  new_biller_id?: string | null;
  new_biller_name?: string | null;
  registrar?: string | null;
  expires_on?: string | null;
  auto_renew?: boolean | null;
  cancel_flag?: boolean | null;
  notes?: string | null;
  updated_at?: string | null;
};

type UploadSummary = {
  inserted: number;
  updated: number;
  skipped: number;
  warnings: string[];
};

export default function DomainRegistryListPage() {
  const router = useRouter();
  const [rows, setRows] = useState<Row[]>([]);
  const [loading, setLoading] = useState(true);
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(50);
  const [total, setTotal] = useState(0);

  // filters
  const [search, setSearch] = useState("");
  const [company, setCompany] = useState("");
  const [companyOpt, setCompanyOpt] = useState<any>(null);
  const [expBefore, setExpBefore] = useState("");
  const [onlyAutoRenew, setOnlyAutoRenew] = useState<"all" | "yes" | "no">(
    "all"
  );
  const [onlyCancel, setOnlyCancel] = useState<"all" | "yes" | "no">("all");

  const [sortField, setSortField] = useState<
    | "domain_name"
    | "expires_on"
    | "company"
    | "new_biller"
    | "auto_renew"
    | "cancel_flag"
    | "updated_at"
  >("domain_name");
  const [sortDir, setSortDir] = useState<"asc" | "desc">("asc");

  const [uploadReport, setUploadReport] = useState<UploadSummary | null>(null);
  const fileRef = useRef<HTMLInputElement | null>(null);

  const loadCompanyOptions = useCallback(async (input: string) => {
    const r = await ApiCall.domainRegistry.companyOptions({
      kind: "own",
      q: input || "",
    });
    return (r.data?.items ?? []).map((o: any) => ({
      value: o.value,
      label: o.label,
    }));
  }, []);

  // -------------------------
  // FETCH (params object)
  // -------------------------
  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const params = {
        page,
        pageSize,
        search: search.trim() || undefined,
        company: company.trim() || undefined,
        expBefore: expBefore.trim() || undefined,
        autoRenew: onlyAutoRenew !== "all" ? onlyAutoRenew : undefined,
        cancel: onlyCancel !== "all" ? onlyCancel : undefined,
        sortField,
        sortDir,
      };

      // Preferred: axios .get(url, { params })
      const res = await ApiCall.domainRegistry.getList(params);

      // If your helper REQUIRES a query string, comment line above and uncomment below:
      // const p = new URLSearchParams(Object.entries(params).reduce((acc, [k, v]) => {
      //   if (v !== undefined && v !== null) acc[k] = String(v);
      //   return acc;
      // }, {} as Record<string, string>));
      // const res = await ApiCall.domainRegistry.getList(`?${p.toString()}`);

      setRows(res.data?.tableData ?? []);
      setTotal(res.data?.totalRecords ?? 0);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Lista betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  }, [
    page,
    pageSize,
    search,
    company,
    expBefore,
    onlyAutoRenew,
    onlyCancel,
    sortField,
    sortDir,
  ]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  // EXPORT (use the SAME params)
  const buildParams = useCallback(() => {
    const params = {
      page,
      pageSize,
      search: search.trim() || undefined,
      company: company.trim() || undefined,
      expBefore: expBefore.trim() || undefined,
      autoRenew: onlyAutoRenew !== "all" ? onlyAutoRenew : undefined,
      cancel: onlyCancel !== "all" ? onlyCancel : undefined,
      sortField,
      sortDir,
    };
    const p = new URLSearchParams(
      Object.entries(params).reduce((acc, [k, v]) => {
        if (v !== undefined && v !== null) acc[k] = String(v);
        return acc;
      }, {} as Record<string, string>)
    );
    return `?${p.toString()}`;
  }, [
    page,
    pageSize,
    search,
    company,
    expBefore,
    onlyAutoRenew,
    onlyCancel,
    sortField,
    sortDir,
  ]);

  const handleExportCsv = async () => {
    try {
      const res = await ApiCall.domainRegistry.exportCsv(buildParams());
      const blob = new Blob([res.data], { type: "text/csv;charset=utf-8" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = "domain_registry.csv";
      a.click();
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("CSV export sikertelen");
    }
  };

  const handleExportXlsx = async () => {
    try {
      const res = await ApiCall.domainRegistry.exportXlsx(buildParams());
      const blob = new Blob([res.data], {
        type: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
      });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = "domain_registry.xlsx";
      a.click();
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error(e);
      toast.error("XLSX export sikertelen");
    }
  };

  // Upload
  const onFileClick = () => {
    if (!fileRef.current) return;
    fileRef.current.value = "";
    fileRef.current.click();
  };
  const onFileChange: React.ChangeEventHandler<HTMLInputElement> = (e) => {
    const f = e.target.files?.[0];
    if (!f) return;
    handleUpload(f);
  };
  const handleUpload = async (file: File) => {
    if (!file) return;
    const form = new FormData();
    form.append("file", file);
    try {
      const res = await ApiCall.domainRegistry.upload(form);
      const summary: UploadSummary | undefined = res?.data?.summary;
      if (summary) {
        setUploadReport(summary);
        toast.success(
          `Import kész • Új: ${summary.inserted}, Frissítve: ${summary.updated}, Kihagyva: ${summary.skipped}`
        );
      } else {
        setUploadReport(null);
        toast.success("Import kész");
      }
      fetchData();
    } catch (e: any) {
      console.error(e);
      const msg =
        e?.response?.data?.error ||
        e?.response?.data?.details ||
        "Feltöltés sikertelen";
      toast.error(msg);
    }
  };

  // helpers
  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  // columns
  const columns = useMemo(
    () => [
      {
        id: "domain_name",
        name: "Domain",
        selector: (r: Row) => r.domain_name,
        cell: (r: Row) => ellipsis(r.domain_name),
        sortable: true,
        width: "260px",
      },
      {
        id: "current_company",
        name: "Jelenlegi cég",
        selector: (r: Row) => r.current_company_id ?? "",
        cell: (r: Row) => ellipsis(r.current_company_name),
        sortable: true,
        width: "220px",
      },
      {
        id: "new_biller",
        name: "Új számlafizető",
        selector: (r: Row) => r.new_biller_id ?? "",
        cell: (r: Row) => ellipsis(r.new_biller_name),
        sortable: true,
        width: "220px",
      },
      {
        id: "registrar",
        name: "Regisztrátor",
        selector: (r: Row) => r.registrar ?? "",
        cell: (r: Row) => ellipsis(r.registrar),
        sortable: true,
        width: "180px",
      },
      {
        id: "expires_on",
        name: "Lejár",
        selector: (r: Row) => r.expires_on ?? "",
        cell: (r: Row) => ellipsis(r.expires_on),
        sortable: true,
        width: "120px",
      },
      {
        id: "auto_renew",
        name: "Auto. megújítás",
        selector: (r: Row) => (r.auto_renew ? "Igen" : "Nem"),
        cell: (r: Row) => ellipsis(r.auto_renew ? "Igen" : "Nem"),
        sortable: true,
        width: "140px",
      },
      {
        id: "cancel_flag",
        name: "Megszüntetni",
        selector: (r: Row) => (r.cancel_flag ? "Igen" : "Nem"),
        cell: (r: Row) => ellipsis(r.cancel_flag ? "Igen" : "Nem"),
        sortable: true,
        width: "140px",
      },
      {
        id: "updated_at",
        name: "Módosítva",
        selector: (r: Row) => r.updated_at ?? "",
        cell: (r: Row) => (
          <div style={{ textAlign: "right" }}>
            {ellipsis(r.updated_at ?? "")}
          </div>
        ),
        sortable: true,
        width: "160px",
      },
    ],
    []
  );

  // expanded rows
  const [expandedRow, setExpandedRow] = useState<Row | null>(null);
  const expandedRowRef = useRef<HTMLDivElement | null>(null);
  const renderExpanded = useCallback(
    (row: Row, onSaved: () => void) => (
      <div ref={expandedRowRef as any} className="p-3">
        <ExpandedDomainRegistry
          id={row.id}
          onSaved={() => {
            onSaved?.();
            fetchData(); // uses CURRENT params (page/pageSize)
          }}
          onDeleted={() => {
            setExpandedRow(null);
            fetchData();
          }}
        />
      </div>
    ),
    [fetchData]
  );

  return (
    <main className="p-4 space-y-4">
      {/* Controls */}
      <div className="flex flex-wrap gap-2 items-center">
        <input
          className="border rounded px-3 py-2"
          placeholder="Keresés (domain, cég, számlafizető)"
          value={search}
          onChange={(e) => {
            setSearch(e.target.value);
            setPage(1);
          }}
        />
        <div className="min-w-60">
          <AsyncSelect
            cacheOptions
            defaultOptions
            loadOptions={loadCompanyOptions}
            value={companyOpt}
            onChange={(opt: any) => {
              setCompanyOpt(opt);
              setCompany(opt?.label || ""); // BE expects name string
              setPage(1);
            }}
            isClearable
            placeholder="Cég / számlafizető"
          />
        </div>
        <input
          type="date"
          className="border rounded px-3 py-2"
          value={expBefore}
          onChange={(e) => {
            setExpBefore(e.target.value);
            setPage(1);
          }}
          title="Lejár előtt (YYYY-MM-DD)"
        />
        <select
          className="border rounded px-3 py-2"
          value={onlyAutoRenew}
          onChange={(e) => {
            setOnlyAutoRenew(e.target.value as any);
            setPage(1);
          }}>
          <option value="all">Autom. megújítás: mind</option>
          <option value="yes">Igen</option>
          <option value="no">Nem</option>
        </select>
        <select
          className="border rounded px-3 py-2"
          value={onlyCancel}
          onChange={(e) => {
            setOnlyCancel(e.target.value as any);
            setPage(1);
          }}>
          <option value="all">Megszüntetni: mind</option>
          <option value="yes">Igen</option>
          <option value="no">Nem</option>
        </select>

        <div className="flex gap-2 ml-auto">
          <button
            onClick={() => router.push("/admin/domain-registry/create")}
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded">
            Új domain
          </button>
          <button
            onClick={handleExportCsv}
            className="px-3 py-2 bg-blue-500/80 hover:bg-blue-500 text-white rounded">
            CSV export
          </button>
          <button
            onClick={handleExportXlsx}
            className="px-3 py-2 bg-green-600/80 hover:bg-green-600 text-white rounded">
            XLSX export
          </button>
          <button
            type="button"
            onClick={() => {
              if (!fileRef.current) return;
              fileRef.current.value = "";
              fileRef.current.click();
            }}
            className="px-3 py-2 bg-gray-700/80 hover:bg-gray-700 text-white rounded">
            Feltöltés (XLSX)
          </button>
          <input
            ref={fileRef}
            type="file"
            accept=".xlsx,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
            onChange={(e) => {
              const f = e.target.files?.[0];
              if (f) {
                const form = new FormData();
                form.append("file", f);
                handleUpload(f);
              }
            }}
            className="sr-only"
          />
        </div>
      </div>

      {uploadReport && (
        <div className="rounded-md border border-amber-300 bg-amber-50 p-3">
          {/* … unchanged upload summary UI … */}
          {/* Keep your existing summary block here */}
        </div>
      )}

      {/* Table */}
      <DataTableWrapper
        keyField="id"
        tableTitle="Domain nyilvántartás"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        /* pagination (server) */
        paginationServer
        paginationCurrentPage={page}
        paginationPerPage={pageSize}
        paginationTotalRows={total}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        /* sorting (server) */
        sortServer
        defaultSortFieldId="domain_name" // keep constant
        onSort={(col: any, dir: "asc" | "desc") => {
          setSortField((col?.id as any) ?? "domain_name");
          setSortDir(dir);
          setPage(1);
        }}
        /* expansion */
        expandableRows
        expandableRowExpanded={(r: Row) => r?.id === expandedRow?.id}
        onRowExpandToggled={(open: boolean, row: Row) => {
          if (open) {
            if (expandedRow && expandedRow.id !== row.id) setExpandedRow(null);
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: Row) =>
          renderExpanded(row, () => {
            toast.success("Mentve");
            fetchData(); // uses *current* params
          })
        }
      />
    </main>
  );
}
