"use client";

import { useMemo, useEffect, useState, useRef, useCallback } from "react";
import dynamic from "next/dynamic";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import { useRouter } from "next/navigation";

type RegistrarRow = {
  id: string;
  name: string;
  email?: string | null;
  url?: string | null;
  created_at?: string;
  updated_at?: string;
};

const ExpandedRegistrar = dynamic(
  () => import("@/components/domain-registrars/ExpandedDomainRegistrar"),
  {
    ssr: false,
    loading: () => <div className="p-3 text-sm text-gray-500">Betöltés…</div>,
  }
);

export default function RegistrarListPage() {
  const router = useRouter();
  const [rows, setRows] = useState<RegistrarRow[]>([]);
  const [total, setTotal] = useState(0);
  const [loading, setLoading] = useState(true);

  // table state
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(50);
  const [q, setQ] = useState("");
  const [sortField, setSortField] = useState<
    "name" | "email" | "url" | "updated_at" | "created_at"
  >("name");
  const [sortDir, setSortDir] = useState<"asc" | "desc">("asc");

  const queryString = useMemo(() => {
    const p = new URLSearchParams();
    p.set("page", String(page));
    p.set("pageSize", String(pageSize));
    if (q.trim()) p.set("q", q.trim());
    p.set("sortField", sortField);
    p.set("sortDir", sortDir);
    return `?${p.toString()}`;
  }, [page, pageSize, q, sortField, sortDir]);

  const fetchData = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.domainRegistrar.getList(queryString);
      setRows(res.data?.items ?? []);
      setTotal(res.data?.totalRecords ?? 0);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Lista betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [queryString]);

  const ellipsis = useCallback((text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  }, []);

  const [expandedRow, setExpandedRow] = useState<RegistrarRow | null>(null);
  const renderExpanded = useCallback(
    (row: RegistrarRow, onSaved: () => void) => (
      <div className="p-3">
        <ExpandedRegistrar
          id={row.id}
          onSaved={() => {
            onSaved?.();
            fetchData();
          }}
          onDeleted={() => {
            setExpandedRow(null);
            fetchData();
          }}
        />
      </div>
    ),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );

  const columns = useMemo(
    () =>
      [
        {
          id: "name",
          name: "Név",
          selector: (r: RegistrarRow) => r.name,
          cell: (r: RegistrarRow) => ellipsis(r.name),
          sortable: true,
          width: "240px",
        },
        {
          id: "email",
          name: "Email",
          selector: (r: RegistrarRow) => r.email ?? "",
          cell: (r: RegistrarRow) => ellipsis(r.email),
          sortable: true,
          width: "260px",
        },
        {
          id: "url",
          name: "URL",
          selector: (r: RegistrarRow) => r.url ?? "",
          cell: (r: RegistrarRow) =>
            r.url ? (
              <a
                className="text-blue-600 underline underline-offset-2"
                href={r.url}
                target="_blank"
                rel="noreferrer">
                {ellipsis(r.url)}
              </a>
            ) : (
              <span>—</span>
            ),
          sortable: true,
          width: "260px",
        },
        {
          id: "updated_at",
          name: "Módosítva",
          selector: (r: RegistrarRow) => r.updated_at ?? "",
          cell: (r: RegistrarRow) => (
            <div style={{ textAlign: "right" }}>
              {ellipsis(r.updated_at ?? "")}
            </div>
          ),
          sortable: true,
          width: "180px",
        },
      ] as const,
    [ellipsis]
  );

  return (
    <main className="p-4 space-y-4">
      <div className="flex items-center gap-2">
        <input
          className="border rounded px-3 py-2"
          placeholder="Keresés (név, email, url)"
          value={q}
          onChange={(e) => {
            setQ(e.target.value);
            setPage(1);
          }}
        />
        <div className="ml-auto flex gap-2">
          <button
            onClick={() => router.push("/admin/domain-registrars/create")}
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded">
            Új regisztrátor
          </button>
        </div>
      </div>
      <DataTableWrapper
        keyField="id"
        tableTitle="Regisztrátorok"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        /* pagination (server) */
        paginationServer
        paginationCurrentPage={page} // NEW: controlled via wrapper
        paginationPerPage={pageSize}
        paginationTotalRows={total}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        /* sorting (server) */
        sortServer
        defaultSortFieldId="name" // keep constant; RDT reads this on mount
        onSort={(col: any, dir: "asc" | "desc") => {
          setSortField((col?.id as any) ?? "name");
          setSortDir(dir);
          setPage(1);
        }}
        /* expansion */
        expandableRows
        expandableRowExpanded={(r: RegistrarRow) => r?.id === expandedRow?.id}
        onRowExpandToggled={(open: boolean, row: RegistrarRow) => {
          if (open) {
            if (expandedRow && expandedRow.id !== row.id) setExpandedRow(null);
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: RegistrarRow) =>
          renderExpanded(row, () => {
            toast.success("Mentve");
            fetchData(); // single refresh (no inner load)
          })
        }
      />

      {/* <DataTableWrapper
        keyField="id"
        tableTitle="Regisztrátorok"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        paginationServer
        // paginationDefaultPage={page}
        paginationCurrentPage={page}
        paginationPerPage={pageSize}
        paginationTotalRows={total}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        sortServer
        // defaultSortFieldId={sortField}
        sortField={sortField}
        sortDirection={sortDir}
        onSort={(col: any, dir: "asc" | "desc") => {
          setSortField((col?.id as any) ?? "name");
          setSortDir(dir);
          setPage(1);
        }}
        // onSort={(col: any, dir: "asc" | "desc") => {
        //   setSortField((col?.id as any) ?? "name");
        //   setSortDir(dir);
        //   setPage(1);
        // }}
        expandableRows
        expandableRowExpanded={(r: RegistrarRow) => r?.id === expandedRow?.id}
        onRowExpandToggled={(open: boolean, row: RegistrarRow) => {
          if (open) {
            if (expandedRow && expandedRow.id !== row.id) setExpandedRow(null);
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: RegistrarRow) =>
          renderExpanded(row, () => {
            toast.success("Mentve");
            fetchData();
          })
        }
      /> */}
    </main>
  );
}
