"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import dynamic from "next/dynamic";
import toast from "react-hot-toast";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";

const ExpandedDocTemplate = dynamic(
  () => import("@/components/document-template-headers/ExpandedDocTemplate"),
  {
    ssr: false,
    loading: () => <div className="p-3 text-sm text-gray-500">Betöltés…</div>,
  }
);

export type DocTemplateRow = {
  id: string;
  name: string;
  kind: string;
  header_html?: string | null;
  footer_html?: string | null;
  header_image_name?: string | null;
  footer_image_name?: string | null;
  is_active: boolean;
  created_at?: string | null;
  updated_at?: string | null;
};

export default function DocTemplateListPage() {
  const router = useRouter();

  const [allRows, setAllRows] = useState<DocTemplateRow[]>([]);
  const [loading, setLoading] = useState(true);

  // filters
  const [search, setSearch] = useState("");
  const [kindFilter, setKindFilter] = useState<"all" | "faq" | "generic">(
    "all"
  );
  const [onlyActive, setOnlyActive] = useState<"all" | "yes" | "no">("all");

  const [selectedIds, setSelectedIds] = useState<string[]>([]);
  const [clearSelectionToggle, setClearSelectionToggle] = useState(false);

  const handleSelectedRowsChange = (state: {
    selectedRows: DocTemplateRow[];
  }) => {
    setSelectedIds(state.selectedRows.map((r) => r.id));
  };
  // fetch all (doc_templates is usually small, so we can load all & filter client-side)
  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const res = await ApiCall.docTemplates.getList({
        // backend supports ?kind=...
        kind: kindFilter !== "all" ? kindFilter : undefined,
      });
      setAllRows(res.data ?? []);
    } catch (e: any) {
      console.error(e);
      toast.error(
        e?.response?.data?.error ?? "Sablon lista betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }, [kindFilter]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  // helper for ellipsis
  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  // client-side filtered rows
  const rows = useMemo(() => {
    return allRows.filter((r) => {
      const matchesSearch =
        !search.trim() ||
        r.name.toLowerCase().includes(search.toLowerCase()) ||
        r.kind.toLowerCase().includes(search.toLowerCase());

      const matchesActive =
        onlyActive === "all"
          ? true
          : onlyActive === "yes"
          ? r.is_active
          : !r.is_active;

      return matchesSearch && matchesActive;
    });
  }, [allRows, search, onlyActive]);

  // columns
  const columns = useMemo(
    () => [
      {
        id: "name",
        name: "Név",
        selector: (r: DocTemplateRow) => r.name,
        cell: (r: DocTemplateRow) => ellipsis(r.name),
        sortable: true,
        width: "260px",
      },
      {
        id: "kind",
        name: "Típus",
        selector: (r: DocTemplateRow) => r.kind,
        cell: (r: DocTemplateRow) => ellipsis(r.kind),
        sortable: true,
        width: "140px",
      },
      {
        id: "header_image",
        name: "Fejléc",
        selector: (r: DocTemplateRow) => r.header_image_name ?? "",
        cell: (r: DocTemplateRow) =>
          r.header_image_name ? (
            <img
              src={ApiCall.mediaStore.getResized(
                "/8/2/0",
                r.header_image_name,
                { format: "webp" }
              )}
              alt="Fejléc"
              className="h-8 w-auto rounded border"
            />
          ) : (
            <span className="text-xs text-gray-400">Nincs</span>
          ),
        sortable: false,
        width: "200px",
      },
      {
        id: "footer_image",
        name: "Lábléc",
        selector: (r: DocTemplateRow) => r.footer_image_name ?? "",
        cell: (r: DocTemplateRow) =>
          r.footer_image_name ? (
            <img
              src={ApiCall.mediaStore.getResized(
                "/8/1/0",
                r.footer_image_name,
                { format: "webp" }
              )}
              alt="Lábléc"
              className="h-8 w-auto rounded border"
            />
          ) : (
            <span className="text-xs text-gray-400">Nincs</span>
          ),
        sortable: false,
        width: "200px",
      },
      {
        id: "is_active",
        name: "Aktív",
        selector: (r: DocTemplateRow) => (r.is_active ? "Igen" : "Nem"),
        cell: (r: DocTemplateRow) => (
          <span className={r.is_active ? "text-emerald-600" : "text-gray-400"}>
            {r.is_active ? "Igen" : "Nem"}
          </span>
        ),
        sortable: true,
        width: "100px",
      },
      {
        id: "updated_at",
        name: "Módosítva",
        selector: (r: DocTemplateRow) => r.updated_at ?? "",
        cell: (r: DocTemplateRow) => (
          <div style={{ textAlign: "right" }}>
            {ellipsis(r.updated_at ?? "")}
          </div>
        ),
        sortable: true,
        width: "180px",
      },
    ],
    []
  );

  // expanded rows
  const [expandedRow, setExpandedRow] = useState<DocTemplateRow | null>(null);

  const renderExpanded = useCallback(
    (row: DocTemplateRow) => (
      <div className="p-3">
        <ExpandedDocTemplate
          id={row.id}
          onSaved={() => {
            toast.success("Sablon frissítve");
            fetchData();
          }}
        />
      </div>
    ),
    [fetchData]
  );

  return (
    <main className="p-4 space-y-4">
      {/* Controls */}
      <div className="flex flex-wrap gap-2 items-center">
        <input
          className="border rounded px-3 py-2"
          placeholder="Keresés (név, típus)"
          value={search}
          onChange={(e) => setSearch(e.target.value)}
        />

        <select
          className="border rounded px-3 py-2"
          value={kindFilter}
          onChange={(e) => setKindFilter(e.target.value as any)}>
          <option value="all">Típus: mind</option>
          <option value="faq">FAQ</option>
          <option value="generic">Általános</option>
        </select>

        <select
          className="border rounded px-3 py-2"
          value={onlyActive}
          onChange={(e) => setOnlyActive(e.target.value as any)}>
          <option value="all">Aktív: mind</option>
          <option value="yes">Csak aktív</option>
          <option value="no">Csak inaktív</option>
        </select>

        <div className="flex gap-2 ml-auto">
          {selectedIds.length > 0 && (
            <button
              onClick={async () => {
                if (
                  !confirm(
                    `Biztosan törölni szeretnéd a kijelölt ${selectedIds.length} sablont?`
                  )
                ) {
                  return;
                }

                try {
                  const res = await ApiCall.docTemplates.removeMany(
                    selectedIds
                  );
                  const { deleted, blocked, notFound } = res.data || {};

                  if (deleted?.length) {
                    toast.success(`Törölve: ${deleted.length} sablon`);
                  }
                  if (blocked?.length) {
                    toast.error(
                      `Nem törölhető: ${blocked.length} sablon (még használatban vannak FAQ-oknál)`
                    );
                  }
                  if (notFound?.length) {
                    console.warn("not found:", notFound);
                  }

                  setSelectedIds([]);
                  setClearSelectionToggle((prev) => !prev);
                  fetchData();
                } catch (e: any) {
                  console.error(e);
                  toast.error(
                    e?.response?.data?.error ?? "Tömeges törlés sikertelen"
                  );
                }
              }}
              className="px-3 py-2 bg-red-600/80 hover:bg-red-600 text-white rounded">
              Kijelöltek törlése ({selectedIds.length})
            </button>
          )}
          <button
            onClick={() =>
              router.push("/admin/document-template-headers/create")
            }
            className="px-3 py-2 bg-emerald-600/80 hover:bg-emerald-600 text-white rounded">
            Új sablon
          </button>
        </div>
      </div>

      {/* Table */}
      <DataTableWrapper
        keyField="id"
        tableTitle="Dokumentum fej/lábléc sablonok"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        // client-side pagination is enough here
        pagination
        paginationPerPage={25}
        paginationRowsPerPageOptions={[25, 50, 100]}
        // client-side sorting (no sortServer)
        defaultSortFieldId="name"
        expandableRows
        expandableRowExpanded={(r: DocTemplateRow) => r?.id === expandedRow?.id}
        onRowExpandToggled={(open: boolean, row: DocTemplateRow) => {
          if (open) {
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: DocTemplateRow) => renderExpanded(row)}
        selectableRows
        onSelectedRowsChange={handleSelectedRowsChange}
        clearSelectedRows={clearSelectionToggle}
      />
    </main>
  );
}
