"use client";

import { useEffect, useMemo, useState } from "react";
import ApiCall from "@/api/api-calls";
import type { Decision, DecisionStatus } from "@/api/routes/decisions";
import {
  Page,
  Card,
  ToolbarRow,
  TextInput,
  Select,
  SecondaryButton,
  Chip,
  EmptyState,
} from "@/components/ui/CardBits";
import { useRouter } from "next/navigation";

function fmt(iso?: string | null) {
  if (!iso) return "-";
  try {
    return new Date(iso).toLocaleString();
  } catch {
    return iso ?? "-";
  }
}

function tone(s: string) {
  switch (s) {
    case "approved":
      return "good";
    case "rejected":
    case "cancelled":
      return "bad";
    case "deferred":
      return "warn";
    case "draft":
    default:
      return "neutral";
  }
}

const STATUSES: Array<DecisionStatus> = [
  "draft",
  "proposed",
  "approved",
  "rejected",
  "deferred",
  "cancelled",
  "superseded",
];

export default function DecisionsLogPage() {
  const router = useRouter();
  const [items, setItems] = useState<Decision[]>([]);
  const [loading, setLoading] = useState(false);

  const [q, setQ] = useState("");
  const [status, setStatus] = useState<DecisionStatus | "">("");

  async function load() {
    setLoading(true);
    try {
      const r = await ApiCall.decisions.list({
        q: q.trim() || undefined,
        status: status || undefined,
      });
      setItems(r.data.decisions ?? []);
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const stats = useMemo(() => {
    const total = items.length;
    const approved = items.filter((x) => x.status === "approved").length;
    const draft = items.filter((x) => x.status === "draft").length;
    const rejected = items.filter((x) => x.status === "rejected").length;
    return { total, approved, draft, rejected };
  }, [items]);

  return (
    <Page
      title="Decision Log"
      subtitle={`összes: ${stats.total} · approved: ${stats.approved} · draft: ${stats.draft} · rejected: ${stats.rejected}`}
      actions={
        <SecondaryButton onClick={() => void load()} disabled={loading}>
          {loading ? "..." : "Frissít"}
        </SecondaryButton>
      }>
      <Card className="mb-3">
        <ToolbarRow>
          <TextInput
            value={q}
            onChange={(e) => setQ(e.target.value)}
            placeholder="Keresés (cím / case / kód)…"
            style={{ flex: 1, minWidth: 260 }}
            onKeyDown={(e: any) => {
              if (e.key === "Enter") void load();
            }}
          />
          <Select
            value={status}
            onChange={(e) => setStatus(e.target.value as any)}>
            <option value="">Összes státusz</option>
            {STATUSES.map((s) => (
              <option key={s} value={s}>
                {s}
              </option>
            ))}
          </Select>
          <SecondaryButton onClick={() => void load()} disabled={loading}>
            Keres
          </SecondaryButton>
        </ToolbarRow>
      </Card>

      {items.length ? (
        <Card className="p-0 overflow-hidden">
          <div className="overflow-auto">
            <table className="min-w-full text-sm">
              <thead className="sticky top-0 bg-white">
                <tr className="border-b border-slate-200">
                  <th className="text-left px-4 py-3 font-extrabold text-slate-700">
                    Döntés
                  </th>
                  <th className="text-left px-4 py-3 font-extrabold text-slate-700">
                    Státusz
                  </th>
                  <th className="text-left px-4 py-3 font-extrabold text-slate-700">
                    Case
                  </th>
                  <th className="text-left px-4 py-3 font-extrabold text-slate-700">
                    Created
                  </th>
                  <th className="text-left px-4 py-3 font-extrabold text-slate-700">
                    Decided
                  </th>
                </tr>
              </thead>
              <tbody>
                {items.map((d) => (
                  <tr
                    key={d.id}
                    className="border-b border-slate-100 hover:bg-slate-50 cursor-pointer"
                    onClick={() => router.push(`/admin/decisions/${d.id}`)}>
                    <td className="px-4 py-3">
                      <div className="font-semibold text-slate-900">
                        {d.title}
                      </div>
                      <div className="text-xs text-slate-500">
                        {d.code ? `kód: ${d.code} · ` : null}
                        id: {d.id.slice(0, 8)}…
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <Chip tone={tone(d.status) as any}>{d.status}</Chip>
                    </td>
                    <td className="px-4 py-3 text-slate-700">
                      {d.case_id.slice(0, 8)}…
                    </td>
                    <td className="px-4 py-3 text-slate-700">
                      {fmt(d.created_at)}
                    </td>
                    <td className="px-4 py-3 text-slate-700">
                      {fmt(d.decided_at)}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </Card>
      ) : (
        <EmptyState
          title="Nincs találat"
          hint="Próbálj másik keresést vagy státuszt."
        />
      )}
    </Page>
  );
}

// "use client";

// import { useEffect, useState } from "react";
// import ApiCall from "@/api/api-calls";
// import type { Decision, DecisionStatus } from "@/api/routes/decisions";
// import {
//   Card,
//   CardTitle,
//   CardMeta,
//   Chip,
//   GridCards,
//   ToolbarRow,
//   TextInput,
//   Select,
//   SecondaryButton,
//   SectionTitle,
// } from "@/components/ui/CardBits";
// import { useRouter } from "next/navigation";

// function fmtDate(iso?: string | null) {
//   if (!iso) return "-";
//   try {
//     return new Date(iso).toLocaleString();
//   } catch {
//     return iso ?? "-";
//   }
// }

// export default function DecisionsLogPage() {
//   const router = useRouter();
//   const [items, setItems] = useState<Decision[]>([]);
//   const [loading, setLoading] = useState(false);

//   const [q, setQ] = useState("");
//   const [status, setStatus] = useState<DecisionStatus | "">("");

//   async function load() {
//     setLoading(true);
//     try {
//       const r = await ApiCall.decisions.list({
//         q: q.trim() || undefined,
//         status: status || undefined,
//       });
//       setItems(r.data.decisions ?? []);
//     } finally {
//       setLoading(false);
//     }
//   }

//   useEffect(() => {
//     void load();
//   }, []);

//   return (
//     <div style={{ padding: 16 }}>
//       <ToolbarRow>
//         <div style={{ fontWeight: 900, fontSize: 18 }}>Decision Log</div>
//         <SecondaryButton onClick={() => void load()} disabled={loading}>
//           {loading ? "..." : "Frissít"}
//         </SecondaryButton>
//       </ToolbarRow>

//       <SectionTitle>Szűrők</SectionTitle>
//       <Card>
//         <ToolbarRow>
//           <TextInput
//             value={q}
//             onChange={(e) => setQ(e.target.value)}
//             placeholder="Keresés..."
//           />
//           <Select
//             value={status}
//             onChange={(e) => setStatus(e.target.value as any)}>
//             <option value="">Összes</option>
//             <option value="draft">draft</option>
//             <option value="proposed">proposed</option>
//             <option value="approved">approved</option>
//             <option value="rejected">rejected</option>
//             <option value="deferred">deferred</option>
//             <option value="cancelled">cancelled</option>
//             <option value="superseded">superseded</option>
//           </Select>
//           <SecondaryButton onClick={() => void load()} disabled={loading}>
//             Keres
//           </SecondaryButton>
//         </ToolbarRow>
//       </Card>

//       <SectionTitle>Eredmények</SectionTitle>
//       <GridCards>
//         {items.map((d) => (
//           <Card
//             key={d.id}
//             style={{ cursor: "pointer" }}
//             onClick={() => router.push(`/admin/decisions/${d.id}`)}>
//             <CardTitle>{d.title}</CardTitle>
//             <CardMeta>
//               case: {d.case_id.slice(0, 8)}… · státusz: {d.status}
//             </CardMeta>
//             <div
//               style={{
//                 display: "flex",
//                 gap: 8,
//                 flexWrap: "wrap",
//                 marginTop: 10,
//               }}>
//               <Chip>created: {fmtDate(d.created_at)}</Chip>
//               {d.decided_at ? (
//                 <Chip>decided: {fmtDate(d.decided_at)}</Chip>
//               ) : null}
//             </div>
//           </Card>
//         ))}
//       </GridCards>
//     </div>
//   );
// }
