"use client";

import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import dynamic from "next/dynamic";
import toast from "react-hot-toast";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";

const ExpandedCompanySegment = dynamic(
  () => import("@/components/company-segments/ExpandedCompanySegment"),
  {
    ssr: false,
    loading: () => <div className="p-3 text-sm text-gray-500">Betöltés…</div>,
  }
);

export type SegmentRow = {
  _id: string; // matches tableData mapping from BE
  name: string;
  slug: string;
  normalizedName: string;
  companyId?: string | null;
  companyName?: string | null;
  createdAt?: string | null;
  updatedAt?: string | null;
};

export default function CompanySegmentListPage() {
  const [rows, setRows] = useState<SegmentRow[]>([]);
  const [loading, setLoading] = useState(true);

  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(25);
  const [total, setTotal] = useState(0);

  const [search, setSearch] = useState("");

  const [sortField, setSortField] = useState<
    | "name"
    | "slug"
    | "normalizedName"
    | "companyName"
    | "createdAt"
    | "updatedAt"
  >("name");
  const [sortDir, setSortDir] = useState<"asc" | "desc">("asc");

  // expanded rows
  const [expandedRow, setExpandedRow] = useState<SegmentRow | null>(null);
  const expandedRowRef = useRef<HTMLDivElement | null>(null);

  // -------------------------
  // FETCH
  // -------------------------
  const fetchData = useCallback(async () => {
    setLoading(true);
    try {
      const params = {
        page,
        pageSize,
        searchTerm: search.trim() || undefined,
        sortField,
        sortDirection: sortDir,
      };

      // if your ApiCall helper accepts params object:
      const res = await ApiCall.companySegments.getList(params);

      // if it only accepts query strings, use:
      // const p = new URLSearchParams(
      //   Object.entries(params).reduce((acc, [k, v]) => {
      //     if (v !== undefined && v !== null) acc[k] = String(v);
      //     return acc;
      //   }, {} as Record<string, string>)
      // );
      // const res = await ApiCall.companySegments.getList(`?${p.toString()}`);

      setRows(res.data?.tableData ?? []);
      setTotal(res.data?.totalRecords ?? 0);
    } catch (e: any) {
      console.error(e);
      toast.error(e?.response?.data?.error ?? "Lista betöltése sikertelen");
    } finally {
      setLoading(false);
    }
  }, [page, pageSize, search, sortField, sortDir]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  // helper for ellipsis cells
  const ellipsis = (text?: string | null) => {
    const v = String(text ?? "");
    return (
      <div
        title={v}
        style={{
          overflow: "hidden",
          whiteSpace: "nowrap",
          textOverflow: "ellipsis",
        }}>
        {v}
      </div>
    );
  };

  const columns = useMemo(
    () => [
      {
        id: "name",
        name: "Szegmens név",
        selector: (r: SegmentRow) => r.name,
        cell: (r: SegmentRow) => ellipsis(r.name),
        sortable: true,
        width: "220px",
      },
      {
        id: "slug",
        name: "Slug",
        selector: (r: SegmentRow) => r.slug,
        cell: (r: SegmentRow) => ellipsis(r.slug),
        sortable: true,
        width: "180px",
      },
      {
        id: "normalizedName",
        name: "Normalizált név",
        selector: (r: SegmentRow) => r.normalizedName,
        cell: (r: SegmentRow) => ellipsis(r.normalizedName),
        sortable: true,
        width: "220px",
      },
      // {
      //   id: "companyName",
      //   name: "Kapcsolt cég",
      //   selector: (r: SegmentRow) => r.companyName ?? "",
      //   cell: (r: SegmentRow) => ellipsis(r.companyName),
      //   sortable: true,
      //   width: "240px",
      // },
      {
        id: "createdAt",
        name: "Létrehozva",
        selector: (r: SegmentRow) => r.createdAt ?? "",
        cell: (r: SegmentRow) => ellipsis(r.createdAt),
        sortable: true,
        width: "180px",
      },
      {
        id: "updatedAt",
        name: "Módosítva",
        selector: (r: SegmentRow) => r.updatedAt ?? "",
        cell: (r: SegmentRow) => ellipsis(r.updatedAt),
        sortable: true,
        width: "180px",
      },
    ],
    []
  );

  // expanded row renderer (like renderExpanded in example)
  const renderExpanded = useCallback(
    (row: SegmentRow) => (
      <div ref={expandedRowRef as any} className="p-3">
        <ExpandedCompanySegment
          id={row._id}
          onSaved={() => {
            toast.success("Szegmens frissítve");
            fetchData();
          }}
          onDeleted={() => {
            setExpandedRow(null);
            fetchData();
          }}
        />
      </div>
    ),
    [fetchData]
  );

  // subheader: simple search like your OwnCompanyList
  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="segmentSubHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={search}
        onInputChange={(_event, newInputValue) => {
          setSearch(newInputValue);
          setPage(1);
        }}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés (név, slug, normalizált név)"
            type="search"
            variant="outlined"
          />
        )}
      />
    </div>
  );

  return (
    <main className="p-4 space-y-4">
      <DataTableWrapper
        keyField="_id"
        tableTitle="Cégszegmensek"
        data={rows}
        columns={columns as any}
        progressPending={loading}
        progressComponent={
          <div className="p-3 text-sm text-gray-500">Betöltés…</div>
        }
        /* pagination (server) */
        paginationServer
        paginationCurrentPage={page}
        paginationPerPage={pageSize}
        paginationTotalRows={total}
        onChangePage={(p) => setPage(p)}
        onChangeRowsPerPage={(s) => {
          setPageSize(s);
          setPage(1);
        }}
        /* sorting (server) */
        sortServer
        defaultSortFieldId="name"
        onSort={(col: any, dir: "asc" | "desc") => {
          setSortField((col?.id as any) ?? "name");
          setSortDir(dir);
          setPage(1);
        }}
        /* filters / header */
        subHeader
        subHeaderComponent={subHeaderComponent}
        subHeaderAlign="right"
        subHeaderWrap
        tableOptionsPath="/admin/company-segment/create"
        /* expansion */
        expandableRows
        expandableRowExpanded={(r: SegmentRow) => r?._id === expandedRow?._id}
        onRowExpandToggled={(open: boolean, row: SegmentRow) => {
          if (open) {
            if (expandedRow && expandedRow._id !== row._id)
              setExpandedRow(null);
            setExpandedRow(row);
          } else {
            setExpandedRow(null);
          }
        }}
        renderExpandableRows={(row: SegmentRow) => renderExpanded(row)}
      />
    </main>
  );
}
