"use client";

import { useEffect, useMemo, useState } from "react";
import toast from "react-hot-toast";
import ApiCall from "@/api/api-calls";
import type { CaseType } from "@/api/routes/case-types";
import {
  Page,
  Card,
  CardHeader,
  CardBody,
  CardTitle,
  CardMeta,
  ToolbarRow,
  TextInput,
  Button,
  SecondaryButton,
  Select,
  Field,
  Divider,
  Chip,
  EmptyState,
} from "@/components/ui/CardBits";

export default function CaseTypesAdminPage() {
  const [q, setQ] = useState("");
  const [includeInactive, setIncludeInactive] = useState(false);
  const [items, setItems] = useState<CaseType[]>([]);
  const [loading, setLoading] = useState(false);

  // create form
  const [key, setKey] = useState("");
  const [name, setName] = useState("");
  const [desc, setDesc] = useState("");
  const [creating, setCreating] = useState(false);

  async function load() {
    setLoading(true);
    try {
      const r = await ApiCall.caseTypes.list({
        q: q.trim() || undefined,
        includeInactive: includeInactive ? 1 : 0,
      });
      setItems(r.data.caseTypes ?? []);
    } catch (e: any) {
      toast.error(
        e?.response?.data?.message ?? "Case types betöltése sikertelen"
      );
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void load();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const filtered = useMemo(() => {
    const s = q.trim().toLowerCase();
    if (!s) return items;
    return items.filter((x) => `${x.key} ${x.name}`.toLowerCase().includes(s));
  }, [items, q]);

  async function onCreate() {
    if (!key.trim() || !name.trim()) return;
    setCreating(true);
    try {
      await ApiCall.caseTypes.create({
        key: key.trim(),
        name: name.trim(),
        description: desc.trim() || undefined,
      });
      toast.success("Ügytípus létrehozva");
      setKey("");
      setName("");
      setDesc("");
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Létrehozás sikertelen");
    } finally {
      setCreating(false);
    }
  }

  async function toggleActive(it: CaseType) {
    try {
      await ApiCall.caseTypes.update(it.id, { isActive: !it.is_active });
      toast.success("Mentve");
      await load();
    } catch (e: any) {
      toast.error(e?.response?.data?.message ?? "Mentés sikertelen");
    }
  }

  return (
    <Page
      title="Ügytípusok (case_types)"
      subtitle={loading ? "Betöltés…" : `${filtered.length} elem`}
      actions={
        <>
          <SecondaryButton onClick={() => void load()} disabled={loading}>
            {loading ? "..." : "Frissít"}
          </SecondaryButton>
        </>
      }>
      <Card className="mb-3">
        <CardHeader>
          <div>
            <CardTitle>Keresés</CardTitle>
            <CardMeta>kulcs / név</CardMeta>
          </div>
        </CardHeader>
        <CardBody>
          <ToolbarRow>
            <TextInput
              value={q}
              onChange={(e) => setQ(e.target.value)}
              placeholder="Keresés…"
              style={{ flex: 1 }}
            />
            <Select
              value={includeInactive ? "1" : "0"}
              onChange={(e) => setIncludeInactive(e.target.value === "1")}>
              <option value="0">Csak aktív</option>
              <option value="1">Inaktívval együtt</option>
            </Select>
            <Button onClick={() => void load()} disabled={loading}>
              {loading ? "..." : "Keres"}
            </Button>
          </ToolbarRow>
        </CardBody>
      </Card>

      <Card className="mb-3">
        <CardHeader>
          <div>
            <CardTitle>Új ügytípus</CardTitle>
            <CardMeta>katalógus – nem kötelező workflow</CardMeta>
          </div>
        </CardHeader>
        <CardBody>
          <div className="grid gap-3">
            <div className="grid gap-3 md:grid-cols-2">
              <Field label="Key" hint="pl. panasz, klinika, ingatlan">
                <TextInput
                  value={key}
                  onChange={(e) => setKey(e.target.value)}
                />
              </Field>
              <Field label="Név" hint="emberi megnevezés">
                <TextInput
                  value={name}
                  onChange={(e) => setName(e.target.value)}
                />
              </Field>
            </div>
            <Field label="Leírás" hint="opcionális">
              <TextInput
                value={desc}
                onChange={(e) => setDesc(e.target.value)}
              />
            </Field>
            <ToolbarRow>
              <Button
                onClick={() => void onCreate()}
                disabled={creating || !key.trim() || !name.trim()}>
                {creating ? "..." : "Létrehoz"}
              </Button>
            </ToolbarRow>
          </div>
        </CardBody>
      </Card>

      <Divider />

      {filtered.length ? (
        <div className="grid gap-3">
          {filtered.map((it) => (
            <Card key={it.id}>
              <CardHeader>
                <div style={{ minWidth: 0 }}>
                  <CardTitle>{it.name}</CardTitle>
                  <CardMeta>key: {it.key}</CardMeta>
                </div>
                <div className="flex gap-2 flex-wrap">
                  {it.is_active ? (
                    <Chip tone="good">active</Chip>
                  ) : (
                    <Chip tone="warn">inactive</Chip>
                  )}
                </div>
              </CardHeader>
              <CardBody>
                <div className="text-sm opacity-80">
                  {it.description ?? "—"}
                </div>
                <div className="mt-3">
                  <SecondaryButton onClick={() => void toggleActive(it)}>
                    {it.is_active ? "Inaktiválás" : "Aktiválás"}
                  </SecondaryButton>
                </div>
              </CardBody>
            </Card>
          ))}
        </div>
      ) : (
        <EmptyState title="Nincs elem" hint="Adj hozzá új ügytípust fent." />
      )}
    </Page>
  );
}
