"use client";

import { useRef, useState } from "react";
import ApiCall from "@/api/api-calls";
import toast from "react-hot-toast";

const UploadPage = () => {
  const fileInputRef = useRef(null);
  const [uploadStatus, setUploadStatus] = useState(null);
  const [summary, setSummary] = useState(null);
  const [rows, setRows] = useState([]);
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setUploadStatus("Feltöltés folyamatban…");
    setSummary(null);
    setRows([]);
    setLoading(true);

    const file = fileInputRef.current?.files?.[0];
    if (!file) {
      setUploadStatus("Nincs fájl kiválasztva");
      setLoading(false);
      return;
    }

    const formData = new FormData();
    formData.append("file", file);

    try {
      const res = await ApiCall.inventory.upload(formData);
      // Log once to verify shape:
      console.log("Upload response data:", res.data);

      const data = res.data || {};

      const success = typeof data.success === "boolean" ? data.success : true; // old impl had no flag

      const message = data.message || "Upload complete";

      const effectiveSummary = data.summary || {
        inserted: data.inserted ?? 0,
        updated: data.updated ?? 0,
        skipped: data.skipped ?? 0,
      };

      const serverRows = Array.isArray(data.rows) ? data.rows : [];

      if (!success) {
        const errorMsg =
          message || "❌ A feltöltés nem sikerült (success = false).";
        setUploadStatus(errorMsg);
        toast.error(errorMsg);
        setLoading(false);
        return;
      }

      setSummary(effectiveSummary);
      setRows(serverRows);

      const statusMsg = `${message} – új: ${effectiveSummary.inserted}, frissítve: ${effectiveSummary.updated}, kihagyva: ${effectiveSummary.skipped}`;
      setUploadStatus(statusMsg);
      toast.success(statusMsg);
    } catch (error) {
      console.error("Upload error:", error);
      const msg =
        error?.response?.data?.message ||
        error?.response?.data?.error ||
        "❌ A feltöltés közben hiba történt.";
      setUploadStatus(msg);
      toast.error(msg);
    } finally {
      setLoading(false);
    }
  };

  const renderChange = (c, idx) => {
    const field = c.field;
    const oldVal = c.old === null || c.old === undefined ? "∅" : String(c.old);
    const newVal = c.new === null || c.new === undefined ? "∅" : String(c.new);

    return (
      <li key={idx}>
        <code>{field}</code>:{" "}
        <span className="line-through text-gray-500">{oldVal}</span> →{" "}
        <span className="font-semibold">{newVal}</span>
      </li>
    );
  };

  return (
    <main className="max-w-5xl mx-auto py-10 px-4 space-y-6">
      <header>
        <h1 className="text-2xl font-bold mb-1">Leltár feltöltés</h1>
        <p className="text-sm text-red-600 py-1">
          * Elfogadott formátum: xls/xlsx
        </p>
        <p className="text-sm text-red-600 py-1">
          ** Kötelező mezők: Eszköz név (amennyiben nincs megadva, a megadott
          sor kimarad a feltöltésből)
        </p>
        <ul>
          <li className="text-sm text-gray-600 py-1">
            - Töltsd fel az exportált leltár XLSX-et. A rendszer{" "}
            <code>EXTERNAL_ID</code> alapján frissíti a meglévő eszközöket, vagy
            újakat hoz létre, ha az EXTERNAL_ID oszlop cella értéke üres. A
            feltöltés után látod, mely sorok mely mezőket módosították (pl.{" "}
            <code>Bérlemény?</code>, <code>Eszköz név</code>, stb.).
          </li>
          <li className="text-sm text-gray-600 py-1">
            - Automatikus leltári szám generálása akkor történik meg, ha a
            Beszerző cég vagy annak kódja + az Eszköz csoport megadásra kerül.
            Minden egyéb esetben kézzel kell generálni leltári számot.
          </li>{" "}
        </ul>
      </header>

      <form onSubmit={handleSubmit} className="space-y-4 border rounded p-4">
        <div>
          <input
            type="file"
            name="file"
            accept=".xlsx"
            ref={fileInputRef}
            className="block w-full border rounded px-3 py-2"
          />
        </div>

        <button
          type="submit"
          disabled={loading}
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 disabled:opacity-50">
          {loading ? "Feltöltés…" : "Feltöltés"}
        </button>

        {uploadStatus && (
          <p className="mt-2 text-sm whitespace-pre-wrap">{uploadStatus}</p>
        )}
      </form>

      {summary && (
        <section className="border rounded p-4 bg-gray-50 text-sm">
          <h2 className="font-semibold mb-3">Összegzés</h2>
          <div className="flex flex-wrap gap-4">
            <div>
              ✅ Új eszközök: <b>{summary.inserted}</b>
            </div>
            <div>
              ♻️ Frissített eszközök: <b>{summary.updated}</b>
            </div>
            <div>
              ⚠️ Kihagyott sorok: <b>{summary.skipped}</b>
            </div>
          </div>

          {summary.updated > 0 && rows.length === 0 && (
            <p className="mt-2 text-xs text-orange-600">
              ⚠️ A szerver nem küldött soronkénti változás-listát (
              <code>rows</code>). Ha részletes diffet szeretnél látni,
              ellenőrizd, hogy az API valóban az új{" "}
              <code>uploadAssetDocument</code> implementációt használja.
            </p>
          )}
        </section>
      )}

      {rows && rows.length > 0 && (
        <section className="border rounded p-4 bg-white text-xs">
          <h2 className="font-semibold mb-3">Soronkénti változások</h2>
          <div className="max-h-[32rem] overflow-auto">
            <table className="w-full border-collapse">
              <thead className="bg-gray-100 sticky top-0">
                <tr>
                  <th className="px-2 py-1 text-left border-b">Excel sor</th>
                  <th className="px-2 py-1 text-left border-b">Művelet</th>
                  <th className="px-2 py-1 text-left border-b">
                    Azonosító / kulcs
                  </th>
                  <th className="px-2 py-1 text-left border-b">Változások</th>
                  <th className="px-2 py-1 text-left border-b">Hiba</th>
                </tr>
              </thead>
              <tbody>
                {rows.map((r, idx) => (
                  <tr
                    key={idx}
                    className={`border-b align-top ${
                      r.action === "error"
                        ? "bg-red-50"
                        : r.action === "inserted"
                        ? "bg-green-50"
                        : ""
                    }`}>
                    <td className="px-2 py-1 whitespace-nowrap">
                      {r.rowNumber}
                    </td>
                    <td className="px-2 py-1 whitespace-nowrap">
                      {r.action === "inserted"
                        ? "Új"
                        : r.action === "updated"
                        ? "Frissítés"
                        : r.action === "error"
                        ? "Hiba"
                        : "Kihagyva"}
                    </td>
                    <td className="px-2 py-1">
                      <div className="space-y-1">
                        {r.key?.inventory_number && (
                          <div>
                            <span className="font-semibold">Leltári szám:</span>{" "}
                            {r.key.inventory_number}
                          </div>
                        )}
                        {r.key?.name && (
                          <div>
                            <span className="font-semibold">Név:</span>{" "}
                            {r.key.name}
                          </div>
                        )}
                        {r.externalId && (
                          <div>
                            <span className="font-semibold">EXTERNAL_ID:</span>{" "}
                            <code>{r.externalId}</code>
                          </div>
                        )}
                      </div>
                    </td>
                    <td className="px-2 py-1">
                      {r.changes && r.changes.length > 0 ? (
                        <ul className="space-y-1">
                          {r.changes.map((c, j) => renderChange(c, j))}
                        </ul>
                      ) : (
                        <span className="text-gray-400">Nincs változás</span>
                      )}
                    </td>
                    <td className="px-2 py-1 text-red-600">{r.error || ""}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>

          <p className="mt-2 text-[11px] text-gray-500">
            Ha például a <b>Bérlemény?</b> mezőt módosítod, itt kell látnod,
            hogy <code>is_leased</code> értéke valóban <code>false</code>-ról{" "}
            <code>true</code>-ra váltott. Ha nem változik, vagy hiba van a
            sorban, az is itt jelenik meg.
          </p>
        </section>
      )}
    </main>
  );
};

export default UploadPage;
