"use client";
import React, { useCallback, useEffect, useState } from "react";
import { useParams, useRouter } from "next/navigation";
import ApiCall from "@/api/api-calls";
import {
  VALIDATOR_REQUIRE,
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
} from "@/utils/validators";
import Input from "@/components/shared/input/Input";
import { useForm } from "@/hooks/custom-form-hook";
import Select from "react-select";
import UniversalIconPicker from "@/components/shared/iconPicker/UniversalIconPicker";
import * as FaIcons from "react-icons/fa";
import * as MdIcons from "react-icons/md";
import * as HiIcons from "react-icons/hi";
import * as BsIcons from "react-icons/bs";
import * as AiIcons from "react-icons/ai";
import toast from "react-hot-toast";
import { useAdminMenu } from "@/hooks/useAdminMenu";
// add more as needed

const iconLibs = {
  fa: FaIcons,
  md: MdIcons,
  hi: HiIcons,
  bs: BsIcons,
};

const AdminMenuEdit = () => {
  const params = useParams();
  const router = useRouter();
  const [menuCategoryData, setMenuCategoryData] = useState(null);
  const [rightsList, setRightsList] = useState(null);
  const [selectedRight, setSelectedRight] = useState(null);
  const [selectedIcon, setSelectedIcon] = useState(null);
  const [formState, handleInput, handleSetInputData] = useForm([], false);
  const [parentMenus, setParentMenus] = useState([]);
  const [selectedParent, setSelectedParent] = useState(null);
  const { refreshMenu } = useAdminMenu();

  useEffect(() => {
    const fetchData = async () => {
      const response = params?.id
        ? await ApiCall.adminMenu.getById(params.id)
        : await ApiCall.adminMenu.getCreate();

      if (response.status === 200) {
        const body = response?.data?.body;
        setMenuCategoryData(body?.data);
        setRightsList(body?.rights);
        setParentMenus(body?.parentMenus || []);

        setSelectedIcon(body?.data?.icon || "");
        setSelectedRight(body?.selected ?? null);
        setSelectedParent(
          body?.data?.parentId
            ? body.parentMenus?.find((m) => m.id === body.data.parentId)
            : null
        );
      }
    };

    fetchData();
  }, [params]);

  const handleSubmitMenu = useCallback(async () => {
    const ret = {
      icon: selectedIcon,
      selectedRight: selectedRight,
      adminMenuTitle: formState.inputs.adminMenuTitle,
      adminMenuLink: formState.inputs.adminMenuLink,
      parentId: selectedParent?.value || null,
    };

    const sendData = params?.id
      ? await ApiCall.adminMenu.update(params.id, ret)
      : await ApiCall.adminMenu.postCreate(ret);

    if (sendData.status === 200) {
      toast.success(sendData?.data?.message || "Sikeres mentés");
      await refreshMenu();
      router.push("/admin/admin-menu/list");
    } else {
      toast.error("Sikertelen mentés" + sendData?.data?.error);
    }
  }, [selectedIcon, selectedRight, formState, router, selectedParent, params]);

  return (
    <div>
      {menuCategoryData && (
        <>
          <Input
            element="input"
            id="adminMenuTitle"
            type="text"
            name="adminMenuTitle"
            placeholder="Admin menü név"
            inputcls="w-full border-2 text-center p-2 my-2 outline-blue-500"
            className="w-100"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem lehet üres"}
            onInput={handleInput}
            initialValid={false}
            initialValue={menuCategoryData?.adminMenuTitle?.value}
          />
          <Input
            element="input"
            id="adminMenuLink"
            type="text"
            name="adminMenuLink"
            placeholder="Admin menü link (pl.: /admin/admin-menu/list)"
            inputcls="w-full border-2 text-center p-2 my-2 outline-blue-500"
            className="w-100"
            validators={[
              VALIDATOR_REQUIRE(),
              VALIDATOR_MINLENGTH(2),
              VALIDATOR_MAXLENGTH(254),
            ]}
            errorText={"Nem lehet üres"}
            onInput={handleInput}
            initialValid={false}
            initialValue={menuCategoryData?.adminMenuLink?.value || ""}
          />
          <Input
            element="input"
            type="text"
            id="adminMenuSlug"
            name="adminMenuSlug"
            placeholder="Admin menü url"
            className="w-100"
            inputcls="w-full border-2 text-center p-2 my-2 bg-gray-100 outline-gray-100"
            errorText={"Not valid url"}
            onInput={handleInput}
            initialValid={true}
            initialValue={menuCategoryData?.adminMenuSlug?.value}
            readOnly
          />
          <Select
            options={rightsList}
            className="w-full"
            menuPosition={"fixed"}
            placeholder={
              "Válassza ki, hogy milyen jogosultság legyen szükséges a menüpont eléréséhez."
            }
            backspaceRemovesValue={false}
            name={"selectedRight"}
            defaultValue={selectedRight}
            onChange={(val, e) => {
              // second parameter shows action triggered
              setSelectedRight(val);
            }}
            required
          />
          <Select
            options={parentMenus.map((m) => ({
              value: m.id,
              label: m.title,
            }))}
            className="w-full my-2"
            menuPosition="fixed"
            placeholder="Válasszon szülő menüt (opcionális)"
            value={selectedParent}
            onChange={(val) => setSelectedParent(val)}
            isClearable
          />
          <div className="w-full flex flex-col justify-center items-center py-4">
            {selectedIcon && (
              <div className="flex items-center gap-2 mb-4">
                <span className="text-sm text-gray-400">
                  Kiválasztott ikon:
                </span>
                {(() => {
                  let FoundIcon = null;

                  for (const [, lib] of Object.entries(iconLibs)) {
                    const Icon = lib?.[selectedIcon];
                    if (Icon) {
                      FoundIcon = Icon;
                      break;
                    }
                  }

                  return FoundIcon ? (
                    <>
                      <FoundIcon className="text-2xl text-teal-400" />
                      <p className="text-sm text-gray-400">{selectedIcon}</p>
                    </>
                  ) : (
                    <span className="text-red-500">Ikon nem található</span>
                  );
                })()}
              </div>
            )}
            <UniversalIconPicker
              value={selectedIcon}
              onSelect={(icon) => setSelectedIcon(icon)}
            />
          </div>
        </>
      )}
      {formState?.isValid && (
        <button
          className="w-full px-4 py-2 bg-green-500/75 hover:bg-green-500 text-white rounded-1"
          onClick={handleSubmitMenu}>
          Mentés
        </button>
      )}
    </div>
  );
};

export default AdminMenuEdit;
