// src/app/(admin)/admin/admin-menu-section/list/page.tsx
"use client";

import { useEffect, useMemo, useState } from "react";
import DataTableWrapper from "@/components/shared/data-table/DataTableWrapper";
import ApiCall from "@/api/api-calls";
import Loader from "@/components/shared/loader/global-loading";
import { TextField } from "@mui/material";
import Autocomplete from "@mui/material/Autocomplete";
import Link from "next/link";
import toast from "react-hot-toast";
import useTableDebounce from "@/hooks/useTableDebounce";
import Button from "@/components/shared/button/Button";
import { Pencil, Trash2 } from "lucide-react";
import { useAdminMenu } from "@/hooks/useAdminMenu";

const SectionList = () => {
  const [data, setData] = useState<any[]>([]);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(true);
  const [inputValue, setInputValue] = useState("");
  const [pageSize, setPageSize] = useState(25);
  const [sortField, setSortField] = useState("sorting");
  const [sortDirection, setSortDirection] = useState<"ASC" | "DESC">("ASC");
  const [totalRecords, setTotalRecords] = useState(0);
  const [selectedRows, setSelectedRows] = useState<any[]>([]);
  const [toggleCleared, setToggleCleared] = useState(false);
  const { refreshMenu } = useAdminMenu();

  const debouncedSearchTerm = useTableDebounce(inputValue, 500);

  const query = `?page=${page}&pageSize=${pageSize}&sortField=${sortField}&sortDirection=${sortDirection}&searchTerm=${debouncedSearchTerm.trim()}`;

  const fetchData = async () => {
    setLoading(true);
    try {
      const res = await ApiCall.adminMenuSection.getAll(query);
      setData(res?.data?.tableData ?? []);
      setTotalRecords(res?.data?.totalRecords ?? 0);
    } catch (err) {
      console.error(err);
      toast.error("Nem sikerült betölteni a listát");
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchData();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [query]);

  const handleSelectAction = async (actionType: string, rowsOrRow: any) => {
    const selected = Array.isArray(rowsOrRow) ? rowsOrRow : [rowsOrRow];
    const ids = selected.map((r) => r._id);

    if (actionType === "delete") {
      const ok = confirm(
        `Biztosan törölni szeretnéd a következő elemeket?\n${ids.join(", ")}`
      );
      if (!ok) return;
      try {
        const res = await ApiCall.adminMenuSection.deleteSections(ids);
        if (res.status === 200) {
          toast.success(res?.data?.message || "Törölve");

          fetchData();
          await refreshMenu();

          setToggleCleared(!toggleCleared);
        } else {
          toast.error(res?.data?.error || "Törlés sikertelen");
        }
      } catch (e) {
        console.error(e);
        toast.error("Törlés sikertelen");
      }
    }

    if (actionType === "edit") {
      if (selected.length !== 1) {
        alert("Csak egy elem szerkeszthető egyszerre.");
        return;
      }
      const id = selected[0]._id;
      window.location.href = `/admin/admin-menu-section/edit/${id}`;
    }
  };

  const handleSortingChange = async (id: string, newValue: string) => {
    const newSorting = parseInt(newValue, 10);
    if (Number.isNaN(newSorting)) return;
    try {
      await ApiCall.adminMenuSection.updateSorting(id, newSorting);
      toast.success("Sorrend frissítve");
      fetchData();
    } catch (err) {
      console.error("Sorting update failed:", err);
      toast.error("Hiba a sorrend frissítésekor");
    }
  };

  const tableSelectOptions = [
    { text: "Szerkesztés", type: "edit", className: "bg-blue-500 text-white" },
    { text: "Törlés", type: "delete", className: "bg-red-500 text-white" },
  ];

  const columns = useMemo(
    () => [
      {
        id: "id",
        name: "ID",
        selector: (row: any) => row._id,
        sortable: false,
      },
      {
        id: "title",
        name: "Szekció neve",
        selector: (row: any) => row.sectionTitle,
        sortable: true,
      },
      {
        id: "sorting",
        name: "Sorrend",
        selector: (row: any) => row.sorting,
        sortable: true,
        cell: (row: any) => (
          <input
            type="number"
            className="w-16 rounded border px-2 py-1 text-sm"
            value={row.sorting}
            onChange={(e) => handleSortingChange(row._id, e.target.value)}
          />
        ),
      },
      {
        id: "updated_at",
        name: "Frissítve",
        selector: (row: any) => row.updatedAt,
        sortable: true,
      },
      {
        name: "Műveletek",
        cell: (row: any) => (
          <div className="flex space-x-2">
            <Button
              href={`/admin/admin-menu-section/edit/${row._id}`}
              type="button"
              variant="info"
              icononly
              rounded="full"
              size="sm"
              aria-label="Edit"
              leftIcon={<Pencil />}
            />
            <Button
              type="button"
              variant="danger"
              icononly
              rounded="full"
              size="sm"
              aria-label="Delete" // ✅ important for screen readers
              onClick={() => handleSelectAction("delete", row)}
              leftIcon={<Trash2 className="w-6 h-6" aria-hidden="true" />}
            />

            {/* <Link
              href={`/admin/admin-menu-section/edit/${row._id}`}
              className="rounded bg-blue-500/75 px-3 py-1 text-sm text-white hover:bg-blue-600">
              Szerkesztés
            </Link>
            <button
              onClick={() => handleSelectAction("delete", row)}
              className="rounded bg-red-500/75 px-3 py-1 text-sm text-white hover:bg-red-600">
              Törlés
            </button> */}
          </div>
        ),
      },
    ],
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );

  const subHeaderComponent = (
    <div className="flex flex-wrap w-full justify-between items-center gap-4">
      <Autocomplete
        freeSolo
        id="subHeaderSearch"
        className="w-72"
        disableClearable
        options={[]}
        inputValue={inputValue}
        onInputChange={(event, newInputValue) => setInputValue(newInputValue)}
        renderInput={(params) => (
          <TextField
            {...params}
            label="Keresés"
            type="search"
            variant="outlined"
          />
        )}
      />
      {selectedRows.length > 0 && (
        <div className="flex items-center gap-2 bg-gray-100 px-4 py-2 border rounded">
          <span className="text-sm text-gray-600">
            {selectedRows.length} kiválasztva
          </span>
          {tableSelectOptions.map((option) => (
            <button
              key={option.text}
              onClick={() => handleSelectAction(option.type, selectedRows)}
              className={`rounded px-3 py-1 text-sm ${option.className}`}>
              {option.text}
            </button>
          ))}
        </div>
      )}
    </div>
  );

  return (
    <DataTableWrapper
      tableTitle="Szekciók"
      data={data}
      columns={columns}
      progressPending={loading}
      paginationDefaultPage={page}
      paginationServer
      onChangePage={(p) => setPage(p)}
      progressComponent={<Loader />}
      paginationTotalRows={totalRecords}
      paginationPerPage={pageSize}
      onSort={(col: any, dir: "asc" | "desc") => {
        setSortField(col?.id || "sorting");
        setSortDirection(dir?.toUpperCase?.() === "DESC" ? "DESC" : "ASC");
        setPage(1);
      }}
      onChangeRowsPerPage={(size) => {
        setPageSize(size);
        setPage(1);
      }}
      onFilterClear={fetchData}
      subHeader
      subHeaderComponent={subHeaderComponent}
      subHeaderAlign="right"
      subHeaderWrap
      tableOptionsPath="admin-menu-section/create"
      selectableRows
      onRowSelected={setSelectedRows}
      toggleCleared={toggleCleared}
    />
  );
};

export default SectionList;
