// src/app/(admin)/admin/admin-menu-section/create/page.tsx
"use client";

import { useEffect, useState, useCallback } from "react";
import { useRouter } from "next/navigation";
import Select from "react-select";
import toast from "react-hot-toast";

import ApiCall from "@/api/api-calls";
import { useForm } from "@/hooks/custom-form-hook";
import Input from "@/components/shared/input/Input";
import UniversalIconPicker from "@/components/shared/iconPicker/UniversalIconPicker";
import {
  VALIDATOR_MINLENGTH,
  VALIDATOR_MAXLENGTH,
  VALIDATOR_REQUIRE,
} from "@/utils/validators";
import { useAdminMenu } from "@/hooks/useAdminMenu";

type Option = { value: string; label: string };

const SectionCreate = () => {
  const router = useRouter();

  const [categoryOptions, setCategoryOptions] = useState<Option[]>([]);
  const [selectedCategories, setSelectedCategories] = useState<Option[]>([]);
  const [selectedIcon, setSelectedIcon] = useState<string>("FaFolder");
  const { refreshMenu } = useAdminMenu();

  const [formState, handleInput] = useForm(
    {
      sectionTitle: { value: "", isValid: false },
      sectionSlug: { value: "", isValid: true },
    },
    false
  );

  useEffect(() => {
    (async () => {
      const res = await ApiCall.adminMenuSection.getCreate();
      if (res?.status !== 200) {
        toast.error("Nem sikerült betölteni a létrehozás űrlapot");
        return;
      }
      const body = res.data?.body || {};
      setCategoryOptions(
        (body.categoryOptions || []).map((c: any) => ({
          value: c.id,
          label: c.title ?? c.slug ?? c.id,
        }))
      );
    })();
  }, []);

  // Auto-slugify
  const titleValue = formState.inputs.sectionTitle?.value ?? "";
  const slugValue = formState.inputs.sectionSlug?.value ?? "";
  useEffect(() => {
    if (!titleValue || slugValue) return;
    const s = titleValue
      .toLowerCase()
      .normalize("NFD")
      .replace(/\p{Diacritic}/gu, "")
      .replace(/[^a-z0-9]+/g, "-")
      .replace(/(^-|-$)/g, "");
    // @ts-ignore
    handleInput("sectionSlug", s, true);
  }, [titleValue, slugValue, handleInput]);

  const handleSubmit = useCallback(async () => {
    const payload = {
      sectionTitle: formState.inputs.sectionTitle, // { value, isValid }
      sectionSlug: formState.inputs.sectionSlug, // { value, isValid }
      icon: selectedIcon,
      selectedCategories: selectedCategories.map((o, i) => ({
        id: o.value,
        sorting: i,
      })),
    };
    const out = await ApiCall.adminMenuSection.postCreate(payload);
    if (out?.status === 200) {
      toast.success(out.data?.message || "Sikeres mentés");
      await refreshMenu();

      router.push("/admin/admin-menu-section/list");
    } else {
      toast.error(out?.data?.error || "Sikertelen mentés");
    }
  }, [formState, selectedIcon, selectedCategories, router]);

  return (
    <div className="max-w-2xl mx-auto space-y-3">
      <h1 className="text-xl font-semibold">Új szekció</h1>

      <Input
        element="input"
        id="sectionTitle"
        name="sectionTitle"
        type="text"
        placeholder="Szekció neve"
        inputcls="w-full border p-2"
        validators={[
          VALIDATOR_REQUIRE(),
          VALIDATOR_MINLENGTH(2),
          VALIDATOR_MAXLENGTH(254),
        ]}
        errorText="Nem lehet üres"
        initialValid={false}
        onInput={handleInput}
        value={titleValue}
      />

      <Input
        element="input"
        id="sectionSlug"
        name="sectionSlug"
        type="text"
        placeholder="Slug (opcionális)"
        inputcls="w-full border p-2"
        validators={[VALIDATOR_MAXLENGTH(254)]}
        errorText="Túl hosszú slug"
        initialValid={true}
        onInput={handleInput}
        value={slugValue}
      />

      <div>
        <label className="block text-sm mb-1">
          Kategóriák ebben a szekcióban
        </label>
        <Select
          options={categoryOptions}
          value={selectedCategories}
          onChange={(opts) => setSelectedCategories((opts as Option[]) || [])}
          isMulti
          className="react-select-container"
          classNamePrefix="react-select"
          placeholder="Válassz kategóriá(ka)t"
        />
      </div>

      <div className="pt-2">
        <label className="block text-sm mb-2">Ikon</label>
        <UniversalIconPicker value={selectedIcon} onSelect={setSelectedIcon} />
      </div>

      <button
        className={`w-full mt-2 px-4 py-2 rounded text-white ${
          formState.isValid ? "bg-green-600 hover:bg-green-700" : "bg-gray-300"
        }`}
        disabled={!formState.isValid}
        onClick={handleSubmit}>
        Mentés
      </button>
    </div>
  );
};

export default SectionCreate;
