import apiClient from "@/lib/apiClient";
import attachmentsApi from "@/api/routes/attachments";

export type Thread = {
  id: string;
  code: string;
  case_id: string;
  title: string;
  description: string | null;
  input_due_at: string | null;
  created_by: string | null;
  created_at: string;
  updated_at: string;

  archived_at?: string | null;
  archived_by?: string | null;
};

export type ThreadCommentType = "comment" | "question" | "proposal" | "stance";

export type ThreadComment = {
  id: string;
  thread_id: string;
  author_id: string | null;
  author_name?: string | null;
  comment_type: ThreadCommentType | string;
  body: string;
  created_at: string;
  updated_at: string;

  archived_at?: string | null;
  archived_by?: string | null;
};

export type ThreadAttachment = {
  id: string;
  original_name: string;
  mime_type: string | null;
  size_bytes: string | number | null;
  uploaded_at: string;
  uploaded_by: string | null;
  downloadUrl: string;
};

export const threadsApi = {
  listByCase: (caseRef: string, params?: { includeArchived?: 0 | 1 }) =>
    apiClient.get<{ threads: Thread[] }>(
      `/api/admin/threads/by-case/${caseRef}`,
      {
        params,
      }
    ),

  create: (
    caseRef: string,
    payload: { title: string; description?: string; inputDueAt?: string | null }
  ) =>
    apiClient.post<{ id: string; code: string }>(
      `/api/admin/threads/by-case/${caseRef}`,
      payload
    ),

  // ✅ kell a ThreadDetailPage-hez
  detail: (threadRef: string) =>
    apiClient.get<{ thread: Thread }>(`/api/admin/threads/${threadRef}`),

  patch: (
    threadRef: string,
    payload: {
      title?: string;
      description?: string | null;
      inputDueAt?: string | null;
    }
  ) => apiClient.patch(`/api/admin/threads/${threadRef}`, payload),

  // ✅ backend router: DELETE /threads/:threadRef
  archive: (threadRef: string) =>
    apiClient.delete(`/api/admin/threads/${threadRef}`),

  // ---------- COMMENTS (backend: /threads/:threadRef/comments) ----------
  listComments: (threadRef: string, params?: { includeArchived?: 0 | 1 }) =>
    apiClient.get<{ comments: ThreadComment[] }>(
      `/api/admin/threads/${threadRef}/comments`,
      { params }
    ),

  createComment: (
    threadRef: string,
    payload: {
      body: string;
      commentType?: ThreadCommentType;
      isPublic?: boolean;
      checklist?: { id: string; text: string; done: boolean }[];
    }
  ) =>
    apiClient.post<{ id: string }>(
      `/api/admin/threads/${threadRef}/comments`,
      payload
    ),

  // ✅ backend router: PATCH /threads/:threadRef/comments/:commentId
  patchComment: (
    threadRef: string,
    commentId: string,
    payload: {
      body?: string;
      commentType?: ThreadCommentType;
      isPublic?: boolean;
      checklist?: { id: string; text: string; done: boolean }[];
    }
  ) =>
    apiClient.patch(
      `/api/admin/threads/${threadRef}/comments/${commentId}`,
      payload
    ),

  // ✅ backend router: DELETE /threads/:threadRef/comments/:commentId
  archiveComment: (threadRef: string, commentId: string) =>
    apiClient.delete(`/api/admin/threads/${threadRef}/comments/${commentId}`),

  // ---------- ATTACHMENTS ----------
  listAttachments: (threadRef: string) =>
    apiClient.get<{ attachments: ThreadAttachment[] }>(
      `/api/admin/threads/${threadRef}/attachments`
    ),

  uploadAttachment: (threadRef: string, file: File) =>
    attachmentsApi.uploadToThread(threadRef, file),
};

export default threadsApi;

// import apiClient from "@/lib/apiClient";
// import attachmentsApi from "@/api/routes/attachments";

// export type Thread = {
//   id: string;
//   code: string;
//   case_id: string;
//   title: string;
//   description: string | null;
//   input_due_at: string | null;
//   created_by: string | null;
//   created_at: string;
//   updated_at: string;

//   archived_at?: string | null;
//   archived_by?: string | null;
// };

// export type ThreadCommentType = "comment" | "question" | "proposal" | "stance";

// export type ThreadComment = {
//   id: string;
//   thread_id: string;
//   author_id: string | null;
//   author_name?: string | null;
//   comment_type: ThreadCommentType | string;
//   body: string;
//   created_at: string;
//   updated_at: string;

//   archived_at?: string | null;
//   archived_by?: string | null;
// };

// export type ThreadAttachment = {
//   id: string;
//   original_name: string;
//   mime_type: string | null;
//   size_bytes: string | number | null;
//   uploaded_at: string;
//   uploaded_by: string | null;
//   downloadUrl: string;
// };
// export const threadsApi = {
//   listByCase: (caseRef: string, params?: { includeArchived?: 0 | 1 }) =>
//     apiClient.get<{ threads: Thread[] }>(
//       `/api/admin/threads/by-case/${caseRef}`,
//       { params }
//     ),

//   create: (
//     caseRef: string,
//     payload: { title: string; description?: string; inputDueAt?: string }
//   ) =>
//     apiClient.post<{ id: string; code: string }>(
//       `/api/admin/threads/by-case/${caseRef}`,
//       payload
//     ),

//   // comments
//   listComments: (threadRef: string) =>
//     apiClient.get<{ comments: ThreadComment[] }>(
//       `/api/admin/threads/${threadRef}/comments`
//     ),

//   createComment: (
//     threadRef: string,
//     payload: { body: string; commentType?: ThreadCommentType }
//   ) =>
//     apiClient.post<{ id: string }>(
//       `/api/admin/threads/${threadRef}/comments`,
//       payload
//     ),

//   // thread update / archive
//   patch: (
//     threadRef: string,
//     payload: {
//       title?: string;
//       description?: string | null;
//       inputDueAt?: string | null;
//     }
//   ) => apiClient.patch(`/api/admin/threads/${threadRef}`, payload),

//   archive: (threadRef: string) =>
//     apiClient.delete(`/api/admin/threads/${threadRef}`),

//   // comment update / archive (a TE routered szerint)
//   patchComment: (
//     threadRef: string,
//     commentId: string,
//     payload: { body?: string; commentType?: ThreadCommentType }
//   ) =>
//     apiClient.patch(
//       `/api/admin/threads/${threadRef}/comments/${commentId}`,
//       payload
//     ),

//   archiveComment: (threadRef: string, commentId: string) =>
//     apiClient.delete(`/api/admin/threads/${threadRef}/comments/${commentId}`),

//   // attachments
//   listAttachments: (threadRef: string) =>
//     apiClient.get<{ attachments: ThreadAttachment[] }>(
//       `/api/admin/threads/${threadRef}/attachments`
//     ),

//   // detail (ehhez kell backend GET)
//   detail: (threadRef: string) =>
//     apiClient.get<{ thread: Thread }>(`/api/admin/threads/${threadRef}`),

//   uploadAttachment: (threadRef: string, file: File) =>
//     attachmentsApi.uploadToThread(threadRef, file),
// };

// // export const threadsApi = {
// //   // ✅ caseRef = uuid OR code (ha a backend router még uuid regex-es, akkor itt csak uuid fog menni)
// //   listByCase: (caseRef: string, params?: { includeArchived?: 0 | 1 }) =>
// //     apiClient.get<{ threads: Thread[] }>(
// //       `/api/admin/threads/by-case/${caseRef}`,
// //       { params }
// //     ),

// //   create: (
// //     caseRef: string,
// //     payload: { title: string; description?: string; inputDueAt?: string }
// //   ) =>
// //     apiClient.post<{ id: string; code: string }>(
// //       `/api/admin/threads/by-case/${caseRef}`,
// //       payload
// //     ),

// //   // ✅ threadRef = uuid OR code (ha a backend router még uuid regex-es: csak uuid)
// //   listComments: (threadRef: string, params?: { includeArchived?: 0 | 1 }) =>
// //     apiClient.get<{ comments: ThreadComment[] }>(
// //       `/api/admin/threads/${threadRef}/comments`,
// //       { params }
// //     ),

// //   createComment: (
// //     threadRef: string,
// //     payload: { body: string; commentType?: ThreadCommentType }
// //   ) =>
// //     apiClient.post<{ id: string }>(
// //       `/api/admin/threads/${threadRef}/comments`,
// //       payload
// //     ),

// //   patchThread: (
// //     threadRef: string,
// //     payload: {
// //       title?: string;
// //       description?: string | null;
// //       inputDueAt?: string | null;
// //     }
// //   ) => apiClient.patch(`/api/admin/threads/${threadRef}`, payload),

// //   archiveThread: (threadRef: string) =>
// //     apiClient.post(`/api/admin/threads/${threadRef}/archive`),

// //   patchComment: (
// //     commentId: string,
// //     payload: { body?: string; commentType?: ThreadCommentType }
// //   ) => apiClient.patch(`/api/admin/thread-comments/${commentId}`, payload),

// //   archiveComment: (commentId: string) =>
// //     apiClient.post(`/api/admin/thread-comments/${commentId}/archive`),

// //   listAttachments: (threadRef: string) =>
// //     apiClient.get<{ attachments: ThreadAttachment[] }>(
// //       `/api/admin/threads/${threadRef}/attachments`
// //     ),

// //   // ------------------------------------------------------------
// //   // ✅ ALIASOK: a page-ek ezeket a neveket keresik
// //   // ------------------------------------------------------------

// //   // page.tsx: threadsApi.patch(...)
// //   patch: (
// //     threadRef: string,
// //     payload: {
// //       title?: string;
// //       description?: string | null;
// //       inputDueAt?: string | null;
// //     }
// //   ) => threadsApi.patchThread(threadRef, payload),

// //   // page.tsx: threadsApi.archive(...)
// //   archive: (threadRef: string) => threadsApi.archiveThread(threadRef),

// //   /**
// //    * page.tsx: threadsApi.detail(...)
// //    * Ha nincs backend GET /threads/:id route, ez 404-et fog dobni,
// //    * de a TS error megszűnik.
// //    * (Ha akarod, adok hozzá backend controllert+route-ot is.)
// //    */
// //   detail: (threadRef: string) =>
// //     apiClient.get<{ thread: Thread }>(`/api/admin/threads/${threadRef}`),

// //   // page.tsx: threadsApi.uploadAttachment(...)
// //   uploadAttachment: (threadRef: string, file: File) =>
// //     attachmentsApi.uploadToThread(threadRef, file),
// // };

// export default threadsApi;

// // import apiClient from "@/lib/apiClient";

// // export type Thread = {
// //   id: string;
// //   code: string;
// //   case_id: string;
// //   title: string;
// //   description: string | null;
// //   input_due_at: string | null;
// //   created_by: string | null;
// //   created_at: string;
// //   updated_at: string;

// //   archived_at?: string | null;
// //   archived_by?: string | null;
// // };

// // export type ThreadComment = {
// //   id: string;
// //   thread_id: string;
// //   author_id: string | null;
// //   author_name?: string | null;
// //   comment_type: "comment" | "question" | "proposal" | "stance" | string;
// //   body: string;
// //   created_at: string;
// //   updated_at: string;

// //   archived_at?: string | null;
// //   archived_by?: string | null;
// // };

// // export type ThreadAttachment = {
// //   id: string;
// //   original_name: string;
// //   mime_type: string | null;
// //   size_bytes: string | number | null;
// //   uploaded_at: string;
// //   uploaded_by: string | null;
// //   downloadUrl: string;
// // };

// // export const threadsApi = {
// //   // ✅ caseRef = uuid OR code
// //   listByCase: (caseRef: string, params?: { includeArchived?: 0 | 1 }) =>
// //     apiClient.get<{ threads: Thread[] }>(
// //       `/api/admin/threads/by-case/${caseRef}`,
// //       { params }
// //     ),

// //   create: (
// //     caseRef: string,
// //     payload: { title: string; description?: string; inputDueAt?: string }
// //   ) =>
// //     apiClient.post<{ id: string; code: string }>(
// //       `/api/admin/threads/by-case/${caseRef}`,
// //       payload
// //     ),

// //   // ✅ threadRef = uuid OR code
// //   listComments: (threadRef: string, params?: { includeArchived?: 0 | 1 }) =>
// //     apiClient.get<{ comments: ThreadComment[] }>(
// //       `/api/admin/threads/${threadRef}/comments`,
// //       { params }
// //     ),

// //   createComment: (
// //     threadRef: string,
// //     payload: {
// //       body: string;
// //       commentType?: "comment" | "question" | "proposal" | "stance";
// //     }
// //   ) =>
// //     apiClient.post<{ id: string }>(
// //       `/api/admin/threads/${threadRef}/comments`,
// //       payload
// //     ),

// //   patchThread: (
// //     threadRef: string,
// //     payload: {
// //       title?: string;
// //       description?: string | null;
// //       inputDueAt?: string | null;
// //     }
// //   ) => apiClient.patch(`/api/admin/threads/${threadRef}`, payload),

// //   archiveThread: (threadRef: string) =>
// //     apiClient.post(`/api/admin/threads/${threadRef}/archive`),

// //   patchComment: (
// //     commentId: string,
// //     payload: {
// //       body?: string;
// //       commentType?: "comment" | "question" | "proposal" | "stance";
// //     }
// //   ) => apiClient.patch(`/api/admin/thread-comments/${commentId}`, payload),

// //   archiveComment: (commentId: string) =>
// //     apiClient.post(`/api/admin/thread-comments/${commentId}/archive`),

// //   listAttachments: (threadRef: string) =>
// //     apiClient.get<{ attachments: ThreadAttachment[] }>(
// //       `/api/admin/threads/${threadRef}/attachments`
// //     ),
// // };

// // export default threadsApi;

// // // // src/routes/threads.ts
// // // import apiClient from "@/lib/apiClient";

// // // export type Thread = {
// // //   id: string;
// // //   case_id: string;
// // //   title: string;
// // //   description: string | null;
// // //   input_due_at: string | null;
// // //   created_by: string | null;
// // //   created_at: string;
// // //   updated_at: string;
// // // };

// // // export type ThreadComment = {
// // //   id: string;
// // //   thread_id: string;
// // //   author_id: string | null;
// // //   comment_type: "comment" | "question" | "proposal" | "stance" | string;
// // //   body: string;
// // //   created_at: string;
// // //   updated_at: string;
// // // };

// // // export type ThreadAttachment = {
// // //   id: string;
// // //   original_name: string;
// // //   mime_type: string | null;
// // //   size_bytes: string | number | null;
// // //   uploaded_at: string;
// // //   uploaded_by: string | null;
// // //   downloadUrl: string; // backend prebuilt
// // // };

// // // export const threadsApi = {
// // //   listByCase: (caseId: string) =>
// // //     apiClient.get<{ threads: Thread[] }>(
// // //       `/api/admin/threads/by-case/${caseId}`
// // //     ),

// // //   create: (
// // //     caseId: string,
// // //     payload: { title: string; description?: string; inputDueAt?: string }
// // //   ) =>
// // //     apiClient.post<{ id: string }>(
// // //       `/api/admin/threads/by-case/${caseId}`,
// // //       payload
// // //     ),

// // //   listComments: (threadId: string) =>
// // //     apiClient.get<{ comments: ThreadComment[] }>(
// // //       `/api/admin/threads/${threadId}/comments`
// // //     ),

// // //   createComment: (
// // //     threadId: string,
// // //     payload: {
// // //       body: string;
// // //       commentType?: "comment" | "question" | "proposal" | "stance";
// // //     }
// // //   ) =>
// // //     apiClient.post<{ id: string }>(
// // //       `/api/admin/threads/${threadId}/comments`,
// // //       payload
// // //     ),

// // //   listAttachments: (threadId: string) =>
// // //     apiClient.get<{ attachments: ThreadAttachment[] }>(
// // //       `/api/admin/threads/${threadId}/attachments`
// // //     ),
// // // };

// // // export default threadsApi;
