import apiClient from "@/lib/apiClient";

export type Meeting = {
  id: string;
  code: string;
  title: string;
  starts_at: string;
  ends_at: string | null;
  notes?: string | null;
  created_by: string | null;
  created_at: string;

  archived_at?: string | null;
  archived_by?: string | null;
};

export type MeetingAttachment = {
  id: string;
  original_name: string;
  mime_type: string | null;
  size_bytes: string | number | null;
  uploaded_at: string;
  uploaded_by: string | null;
  downloadUrl: string;
};

export type AgendaItem = {
  id: string;
  meeting_id: string;
  case_id: string;
  thread_id: string | null;
  title: string;
  sort_order: number;
  case_title?: string;
  thread_title?: string | null;

  archived_at?: string | null;
  archived_by?: string | null;
};

export const meetingsApi = {
  list: (params?: {
    from?: string;
    to?: string;
    q?: string;
    includeArchived?: 0 | 1;
  }) =>
    apiClient.get<{ meetings: Meeting[] }>(`/api/admin/meetings`, { params }),

  create: (payload: {
    title: string;
    startsAt: string;
    endsAt?: string;
    notes?: string;
    agendaItems?: Array<{
      caseId: string;
      threadId?: string;
      title: string;
      sortOrder?: number;
    }>;
  }) =>
    apiClient.post<{ id: string; code: string }>(
      `/api/admin/meetings`,
      payload
    ),

  // ✅ meetingRef = uuid OR code
  detail: (meetingRef: string, params?: { includeArchived?: 0 | 1 }) =>
    apiClient.get<{
      meeting: Meeting;
      agenda: AgendaItem[];
      attachments: MeetingAttachment[];
    }>(`/api/admin/meetings/${meetingRef}`, { params }),

  update: (
    meetingRef: string,
    payload: {
      title?: string;
      startsAt?: string;
      endsAt?: string;
      notes?: string | null;
    }
  ) => apiClient.patch(`/api/admin/meetings/${meetingRef}`, payload),

  addAgendaItem: (
    meetingRef: string,
    payload: {
      caseId: string;
      threadId?: string;
      title: string;
      sortOrder?: number;
    }
  ) =>
    apiClient.post<{ id: string }>(
      `/api/admin/meetings/${meetingRef}/agenda-items`,
      payload
    ),

  // patchAgendaItem: (
  //   meetingRef: string,
  //   agendaItemId: string,
  //   payload: { title?: string; sortOrder?: number; threadId?: string | null }
  // ) =>
  //   apiClient.patch(
  //     `/api/admin/meetings/${meetingRef}/agenda-items/${agendaItemId}`,
  //     payload
  //   ),

  archiveAgendaItem: (meetingRef: string, agendaItemId: string) =>
    apiClient.post(
      `/api/admin/meetings/${meetingRef}/agenda-items/${agendaItemId}/archive`
    ),
  unarchiveAgendaItem: (meetingRef: string, agendaItemId: string) =>
    apiClient.post(
      `/api/admin/meetings/${meetingRef}/agenda-items/${agendaItemId}/unarchive`
    ),

  uploadAttachment: (meetingRef: string, file: File) => {
    const form = new FormData();
    form.append("file", file);
    return apiClient.post<{ id: string }>(
      `/api/admin/attachments/meeting/${meetingRef}`,
      form,
      { headers: { "Content-Type": "multipart/form-data" } }
    );
  },

  createActionItemFromAgenda: (
    meetingRef: string,
    payload: {
      agendaItemId: string;
      title?: string;
      description?: string;
      assignedTo?: string;
      dueAt?: string;
    }
  ) =>
    apiClient.post<{ id: string }>(
      `/api/admin/meetings/${meetingRef}/action-items`,
      payload
    ),

  archive: (meetingRef: string) =>
    apiClient.post(`/api/admin/meetings/${meetingRef}/archive`),
  patchAgendaItem: (
    meetingRef: string,
    agendaItemId: string,
    payload: { title?: string; sortOrder?: number; threadId?: string | null }
  ) =>
    apiClient.patch(
      `/api/admin/meetings/${meetingRef}/agenda-items/${agendaItemId}`,
      payload
    ),
};

export default meetingsApi;

// import apiClient from "@/lib/apiClient";

// export type Meeting = {
//   id: string;
//   title: string;
//   starts_at: string;
//   ends_at: string | null;
//   notes?: string | null;
//   created_by: string | null;
//   created_at: string;
// };

// export type MeetingAttachment = {
//   id: string;
//   original_name: string;
//   mime_type: string | null;
//   size_bytes: string | number | null;
//   uploaded_at: string;
//   uploaded_by: string | null;
//   downloadUrl: string;
// };

// export type AgendaItem = {
//   id: string;
//   meeting_id: string;
//   case_id: string;
//   thread_id: string | null;
//   title: string;
//   sort_order: number;
//   case_title?: string;
//   thread_title?: string | null;

//   // >>> ARCHIVE SUPPORT (optional, backend may or may not send yet)
//   archived_at?: string | null;
//   archived_by?: string | null;
// };

// export const meetingsApi = {
//   list: (params?: { from?: string; to?: string }) =>
//     apiClient.get<{ meetings: Meeting[] }>(`/api/admin/meetings`, { params }),

//   create: (payload: {
//     title: string;
//     startsAt: string; // ISO
//     endsAt?: string; // ISO
//     notes?: string;
//     agendaItems?: Array<{
//       caseId: string;
//       threadId?: string;
//       title: string;
//       sortOrder?: number;
//     }>;
//   }) => apiClient.post<{ id: string }>(`/api/admin/meetings`, payload),

//   // >>> UPDATED: optional includeArchived flag
//   detail: (meetingId: string, params?: { includeArchived?: 0 | 1 }) =>
//     apiClient.get<{
//       meeting: Meeting;
//       agenda: AgendaItem[];
//       attachments: MeetingAttachment[];
//     }>(`/api/admin/meetings/${meetingId}`, { params }),

//   addAgendaItem: (
//     meetingId: string,
//     payload: {
//       caseId: string;
//       threadId?: string;
//       title: string;
//       sortOrder?: number;
//     }
//   ) =>
//     apiClient.post<{ id: string }>(
//       `/api/admin/meetings/${meetingId}/agenda-items`,
//       payload
//     ),

//   patchAgendaItem: (
//     meetingId: string,
//     agendaItemId: string,
//     payload: { title?: string; sortOrder?: number; threadId?: string | null }
//   ) =>
//     apiClient.patch(
//       `/api/admin/meetings/${meetingId}/agenda-items/${agendaItemId}`,
//       payload
//     ),

//   // >>> NOTE: backend oldalon ezt mostantól nyugodtan használhatjátok archive-ra is
//   deleteAgendaItem: (meetingId: string, agendaItemId: string) =>
//     apiClient.delete(
//       `/api/admin/meetings/${meetingId}/agenda-items/${agendaItemId}`
//     ),

//   uploadAttachment: (meetingId: string, file: File) => {
//     const form = new FormData();
//     form.append("file", file);
//     return apiClient.post<{ id: string }>(
//       `/api/admin/attachments/meeting/${meetingId}`,
//       form,
//       { headers: { "Content-Type": "multipart/form-data" } }
//     );
//   },

//   createActionItemFromAgenda: (
//     meetingId: string,
//     payload: {
//       agendaItemId: string;
//       title?: string;
//       description?: string;
//       assignedTo?: string;
//       dueAt?: string;
//     }
//   ) =>
//     apiClient.post<{ id: string }>(
//       `/api/admin/meetings/${meetingId}/action-items`,
//       payload
//     ),
// };

// export default meetingsApi;

// import apiClient from "@/lib/apiClient";

// export type Meeting = {
//   id: string;
//   title: string;
//   starts_at: string;
//   ends_at: string | null;
//   notes?: string | null;
//   created_by: string | null;
//   created_at: string;
// };

// export type MeetingAttachment = {
//   id: string;
//   original_name: string;
//   mime_type: string | null;
//   size_bytes: string | number | null;
//   uploaded_at: string;
//   uploaded_by: string | null;
//   downloadUrl: string;
// };

// export type AgendaItem = {
//   id: string;
//   meeting_id: string;
//   case_id: string;
//   thread_id: string | null;
//   title: string;
//   sort_order: number;
//   case_title?: string;
//   thread_title?: string | null;
// };

// export const meetingsApi = {
//   list: (params?: { from?: string; to?: string }) =>
//     apiClient.get<{ meetings: Meeting[] }>(`/api/admin/meetings`, { params }),

//   create: (payload: {
//     title: string;
//     startsAt: string; // ISO
//     endsAt?: string; // ISO
//     notes?: string;
//     agendaItems?: Array<{
//       caseId: string;
//       threadId?: string;
//       title: string;
//       sortOrder?: number;
//     }>;
//   }) => apiClient.post<{ id: string }>(`/api/admin/meetings`, payload),

//   detail: (meetingId: string) =>
//     apiClient.get<{
//       meeting: Meeting;
//       agenda: AgendaItem[];
//       attachments: MeetingAttachment[];
//     }>(`/api/admin/meetings/${meetingId}`),

//   addAgendaItem: (
//     meetingId: string,
//     payload: {
//       caseId: string;
//       threadId?: string;
//       title: string;
//       sortOrder?: number;
//     }
//   ) =>
//     apiClient.post<{ id: string }>(
//       `/api/admin/meetings/${meetingId}/agenda-items`,
//       payload
//     ),

//   patchAgendaItem: (
//     meetingId: string,
//     agendaItemId: string,
//     payload: { title?: string; sortOrder?: number; threadId?: string | null }
//   ) =>
//     apiClient.patch(
//       `/api/admin/meetings/${meetingId}/agenda-items/${agendaItemId}`,
//       payload
//     ),

//   deleteAgendaItem: (meetingId: string, agendaItemId: string) =>
//     apiClient.delete(
//       `/api/admin/meetings/${meetingId}/agenda-items/${agendaItemId}`
//     ),

//   uploadAttachment: (meetingId: string, file: File) => {
//     const form = new FormData();
//     form.append("file", file);
//     return apiClient.post<{ id: string }>(
//       `/api/admin/attachments/meeting/${meetingId}`,
//       form,
//       { headers: { "Content-Type": "multipart/form-data" } }
//     );
//   },

//   createActionItemFromAgenda: (
//     meetingId: string,
//     payload: {
//       agendaItemId: string;
//       title?: string;
//       description?: string;
//       assignedTo?: string;
//       dueAt?: string;
//     }
//   ) =>
//     apiClient.post<{ id: string }>(
//       `/api/admin/meetings/${meetingId}/action-items`,
//       payload
//     ),
// };

// export default meetingsApi;
