import apiClient from "@/lib/apiClient";
import { wrapApiWithTokenRefresh } from "@/lib/withTokenRefresh";
import { findLastIndexOfChar } from "@/utils/findLastIndexOfChar";

const baseMediaStoreApi = {
  getAll: ({
    page = 1,
    limit = 20,
    search = "",
    category = "",
    tag = "",
    sortBy = "createdAt",
    sortOrder = "desc",
    dateFrom = "",
    dateTo = "",
  }) => {
    const params = new URLSearchParams({
      page: String(page),
      limit: String(limit), // ✅ must use the passed one!
      search,
      category,
      tag,
      sortBy,
      sortOrder,
    });

    if (dateFrom) params.append("dateFrom", dateFrom);
    if (dateTo) params.append("dateTo", dateTo);

    return apiClient.get(`/api/admin/media-store/list?${params.toString()}`);
  },
  getResized: (
    size: string, // e.g. "/600/60/0"
    image?: string,
    options?: {
      format?: "webp" | "png" | "jpeg" | "jpg";
      mode?: "ratio" | "pixel";
    }
  ) => {
    if (!image) {
      return "/images/no-image.png";
    }

    const { format = "webp", mode = "ratio" } = options || {};

    // On the client, prefer NEXT_PUBLIC_ env
    const base = process.env.SERVER_URL || "http://localhost:8110";

    // Use the original filename, do NOT strip extension here
    const encodedFilename = encodeURIComponent(image);

    const params = new URLSearchParams();
    if (format) params.set("format", format);
    if (mode && mode !== "ratio") params.set("mode", mode); // "ratio" is default on server

    const query = params.toString() ? `?${params.toString()}` : "";

    // If your route is actually /api/admin/i/:width/:height/:cropped/:filename,
    // then change `${base}/i${size}/...` to `${base}/api/admin/i${size}/...`
    return `${base}/api/admin/i${size}/${encodedFilename}${query}`;
  },

  // getResized: (size, image) => {
  //   if (typeof image !== "undefined") {
  //     const base = process.env.SERVER_URL || "http://localhost:8110"; // or wherever your API is hosted
  //     const filename = findLastIndexOfChar.before(image, ".");

  //     return `${base}/api/admin/i${size}/${filename}.webp`;
  //   } else {
  //     return "/images/no-image.png";
  //   }
  // },
  editImage: (image) => apiClient.get(`/api/admin/media-store/edit/${image}`),
  search: (query, options) =>
    apiClient.get(`/api/admin/media-store/search${query}`, options),
  upload: (formData, onUploadProgress) => {
    return apiClient.post("/api/admin/media-store/upload", formData, {
      headers: {
        "Content-Type": "multipart/form-data",
      },
      onUploadProgress,
    });
  },
  // editImage: (id, data) =>
  //   apiClient.put(`/api/admin/media-store/edit/${id}`, data),
  deleteImage: (toRemove) =>
    apiClient.delete(`/api/admin/media-store/delete`, {
      data: { ids: toRemove },
    }),
  removeAttachedDocument: (id) =>
    apiClient.delete(`/api/admin/media-store/documents/${id}`),
  getDocumentDownloadUrl: (docId) =>
    `${process.env.SERVER_URL}/api/admin/media-store/documents/${docId}/download`,
};

const mediaStoreApi = wrapApiWithTokenRefresh(baseMediaStoreApi, [
  "getAll",
  "editImage",
  "search",
]);

export default mediaStoreApi;
