// app/api/routes/email-registry.ts
import apiClient from "@/lib/apiClient";
import { wrapApiWithTokenRefresh } from "@/lib/withTokenRefresh";
import type { EmailUpsertPayload } from "@/components/email-registry/EmailRegistryForm";

export type AutoresponderPayload = {
  enabled: boolean;
  fromName: string;
  subject: string;
  body: string;
  isHtml?: boolean;
  intervalHours?: number;
  start?: string | null; // ISO string from <input type="datetime-local">
  stop?: string | null;
};

const baseEmailRegistryApi = {
  getList: (query: string) =>
    apiClient.get(`/api/admin/email-registry/list${query}`),

  upload: (form: FormData) =>
    apiClient.post(`/api/admin/email-registry/upload`, form),

  getTypes: () => apiClient.get("/api/admin/email-registry/types"),
  getGroups: () => apiClient.get("/api/admin/email-registry/groups"),
  getHandlerStatuses: () => apiClient.get("/api/admin/email-registry/statuses"),

  // 🔹 NEW: people & segments option lists
  getHandlerPersons: () =>
    apiClient.get("/api/admin/email-registry/handlers/persons"),
  getHandlerSegments: () =>
    apiClient.get("/api/admin/email-registry/handlers/segments"),

  // 🔹 use the new payload shape for create/update
  create: (payload: EmailUpsertPayload) =>
    apiClient.post(`/api/admin/email-registry`, payload),

  update: (id: string, payload: Partial<EmailUpsertPayload>) =>
    apiClient.put(`/api/admin/email-registry/${id}`, payload),

  remove: (id: string) => apiClient.delete(`/api/admin/email-registry/${id}`),

  getOne: (id: string) => apiClient.get(`/api/admin/email-registry/${id}`),

  exportCsv: (query: string) =>
    apiClient.get(`/api/admin/email-registry/export${query}`, {
      responseType: "blob",
    }),

  exportXlsx: (query: string) =>
    apiClient.get(
      `/api/admin/email-registry/export${query}${query ? "&" : "?"}format=xlsx`,
      { responseType: "blob" }
    ),

  emailOptions: (search: string) =>
    apiClient.get(
      `/api/admin/email-registry/options/emails?search=${encodeURIComponent(
        search
      )}`
    ),

  getPasswordMeta: (id: string) =>
    apiClient.get(`/api/admin/email-registry/${id}/password`),

  getPasswordReveal: (id: string) =>
    apiClient.get(`/api/admin/email-registry/${id}/password?reveal=1`),

  setPassword: (id: string, password: string) =>
    apiClient.put(`/api/admin/email-registry/${id}/password`, { password }),
  updateAccount: (
    domain: string,
    localPart: string,
    payload: {
      newPassword?: string;
      newQuotaMb?: number;
      suspendLogin?: boolean;
      suspendIncoming?: boolean;
      suspendOutgoing?: boolean;
      forwardTo?: string[] | null;
      enableForward?: boolean;
      autoresponder?: AutoresponderPayload; // 🔹 NEW
    }
  ) => {
    return apiClient.put(
      `/api/admin/email-accounts/${encodeURIComponent(
        domain
      )}/${encodeURIComponent(localPart)}`,
      payload
    );
  },
  createAccount: (
    domain: string,
    localPart: string,
    payload: {
      password: string;
      quotaMb?: number;
    }
  ) => {
    return apiClient.post(`/api/admin/email-accounts`, {
      localPart,
      domain,
      password: payload.password,
      quotaMb: payload.quotaMb ?? 1024, // default 1 GB
    });
  },
  emailRecipients: (params: { search?: string }) =>
    apiClient.get("/api/admin/email-recipients/search", { params }),
};

const emailRegistryApi = wrapApiWithTokenRefresh(baseEmailRegistryApi, []);

export default emailRegistryApi;
