import apiClient from "@/lib/apiClient";
import attachmentsApi from "@/api/routes/attachments";

export type DecisionStatus =
  | "draft"
  | "approved"
  | "rejected"
  | "deferred"
  | "cancelled"
  | string;

export type Decision = {
  id: string;
  code: string;

  case_id: string;
  thread_id?: string | null;
  meeting_id?: string | null;

  title: string;
  decision_text?: string | null;

  status: DecisionStatus;
  decided_at?: string | null;

  owner_user_id?: string | null;
  due_at?: string | null;

  created_at: string;
  updated_at?: string;

  archived_at?: string | null;
  archived_by?: string | null;
};

export type DecisionAttachment = {
  id: string;
  original_name: string;
  mime_type: string | null;
  size_bytes: string | number | null;
  uploaded_at: string;
  uploaded_by: string | null;
  downloadUrl: string;
};

export type NoteChecklistItem = { id: string; text: string; done: boolean };

export type DecisionNoteComment = {
  id: string;
  decision_id: string;
  author_id: string | null;
  author_name?: string | null;
  body: string;
  created_at: string;
  updated_at?: string;

  // ✅ notes feature
  is_public: boolean;
  checklist: NoteChecklistItem[];

  archived_at?: string | null;
  archived_by?: string | null;
};

export const decisionsApi = {
  // ✅ caseRef = uuid OR code
  listByCase: (caseRef: string, params?: { includeArchived?: 0 | 1 }) =>
    apiClient.get<{ decisions: Decision[] }>(
      `/api/admin/decisions/by-case/${caseRef}`,
      { params }
    ),

  create: (
    caseRef: string,
    payload: {
      title: string;
      decisionText: string;
      threadId?: string;
      meetingId?: string;
      ownerUserId?: string;
      dueAt?: string;
    }
  ) =>
    apiClient.post<{ id: string; code: string }>(
      `/api/admin/decisions/by-case/${caseRef}`,
      payload
    ),

  updateStatus: (
    decisionRef: string,
    payload: { status: DecisionStatus; decidedAt?: string }
  ) => apiClient.patch(`/api/admin/decisions/${decisionRef}/status`, payload),

  list: (params?: {
    status?: DecisionStatus;
    from?: string;
    to?: string;
    q?: string;
    caseId?: string;
    meetingId?: string;
    includeArchived?: 0 | 1;
  }) =>
    apiClient.get<{ decisions: Decision[] }>(`/api/admin/decisions`, {
      params,
    }),

  // ✅ decisionRef = uuid OR code
  detail: (decisionRef: string) =>
    apiClient.get<{ decision: Decision; attachments: DecisionAttachment[] }>(
      `/api/admin/decisions/${decisionRef}`
    ),
  patch: (
    decisionRef: string,
    payload: {
      title?: string;
      decisionText?: string;
      status?: DecisionStatus;
      dueAt?: string | null;
      ownerUserId?: string | null;
    }
  ) => apiClient.patch(`/api/admin/decisions/${decisionRef}`, payload),

  archive: (decisionRef: string) =>
    apiClient.post(`/api/admin/decisions/${decisionRef}/archive`),
  unarchive: (decisionRef: string) =>
    apiClient.post(`/api/admin/decisions/${decisionRef}/unarchive`),
  uploadAttachment: (decisionRef: string, file: File) =>
    attachmentsApi.uploadToDecision(decisionRef, file),
  listComments: (decisionRef: string, params?: { includeArchived?: 0 | 1 }) =>
    apiClient.get<{ comments: DecisionNoteComment[] }>(
      `/api/admin/decisions/${decisionRef}/comments`,
      { params }
    ),

  createComment: (
    decisionRef: string,
    payload: {
      body: string;
      isPublic?: boolean;
      checklist?: NoteChecklistItem[];
    }
  ) =>
    apiClient.post<{ id: string }>(
      `/api/admin/decisions/${decisionRef}/comments`,
      payload
    ),
  patchComment: (
    decisionRef: string,
    commentId: string,
    payload: {
      body?: string;
      isPublic?: boolean;
      checklist?: NoteChecklistItem[];
    }
  ) =>
    apiClient.patch(
      `/api/admin/decisions/${decisionRef}/comments/${commentId}`,
      payload
    ),
};

export default decisionsApi;
