import apiClient from "@/lib/apiClient";

export type CaseFlowType = "projekt" | "folyamatos";
export type CaseStatus =
  | "new"
  | "in_progress"
  | "blocked"
  | "needs_decision"
  | "closed"
  | "cancelled"
  | "done"
  | "open";

export type CaseListItem = {
  id: string;
  code: string | null;
  title: string;
  description?: string | null;

  case_type?: string | null;
  flow_type?: CaseFlowType | string | null; // backend lehet sima text
  playbook_id?: string | null;

  // ✅ NEW: backend listCases már küldi (JOIN playbooks)
  playbook_key?: string | null;
  playbook_name?: string | null;
  playbook_description?: string | null;

  status?: CaseStatus | string;
  priority?: number | null;

  owner_user_id?: string | null;
  owner_name?: string | null;
  owner_email?: string | null;
  owner_username?: string | null;

  due_at?: string | null;
  created_at: string;
  updated_at?: string | null;

  archived_at?: string | null;
  archived_by?: string | null;
};

// export type CaseListItem = {
//   id: string;
//   code: string | null;
//   title: string;
//   description?: string | null;

//   case_type?: string | null;
//   flow_type?: CaseFlowType | string | null; // backend lehet sima text
//   playbook_id?: string | null;

//   status?: CaseStatus | string;
//   priority?: number | null;

//   owner_user_id?: string | null;
//   owner_name?: string | null;
//   owner_email?: string | null;
//   owner_username?: string | null;

//   due_at?: string | null;
//   created_at: string;
//   updated_at?: string | null;

//   archived_at?: string | null;
//   archived_by?: string | null;
// };

export type CaseMember = {
  user_id: string;
  role: string;
  is_active: boolean;
  joined_at: string;

  user_name?: string | null;
  user_email?: string | null;
  user_username?: string | null;
};

export type CaseDetail = {
  case: CaseListItem;
  members: CaseMember[];
};

// export type CasePlaybookChecklistStep = {
//   id: string;
//   playbook_id: string;
//   sort_order: number;
//   title: string;
//   description?: string | null;

//   is_done: boolean;
//   done_at?: string | null;
//   done_by?: string | null;
//   notes?: string | null;
// };

export type CasePlaybookChecklistResponse = {
  caseId: string;
  playbook: null | {
    id: string;
    key: string;
    name: string;
    description?: string | null;
    default_case_type?: string | null;
    default_flow_type?: string | null;
    is_active: boolean;
  };
  steps: CasePlaybookChecklistStep[];
};

export type CasePlaybookChecklistStep = {
  step_id: string;
  sort_order: number;
  title: string;
  description: string | null;
  is_done: boolean;
  done_at: string | null;
  done_by: string | null;
  notes: string | null;
};

export type CasePlaybookChecklist = {
  caseId: string;
  playbookId: string | null;
  steps: CasePlaybookChecklistStep[];
};

export const casesApi = {
  list: (params?: {
    q?: string;
    mine?: 0 | 1;
    status?: string;
    includeArchived?: 0 | 1;
  }) =>
    apiClient.get<{ cases: CaseListItem[] }>(`/api/admin/cases`, { params }),

  // ✅ NEW: flowType + playbookId
  create: (payload: {
    title: string;
    description?: string;
    caseType?: string;
    flowType?: CaseFlowType;
    playbookId?: string; // uuid
    priority?: number;
    dueAt?: string;
    ownerUser?: string;
  }) =>
    apiClient.post<{ id: string; code: string }>(`/api/admin/cases`, payload),

  detail: (caseRef: string) =>
    apiClient.get<CaseDetail>(`/api/admin/cases/${caseRef}`),

  // ✅ NEW: flowType + playbookId
  update: (
    caseRef: string,
    payload: {
      title?: string;
      description?: string | null;

      caseType?: string | null;
      flowType?: CaseFlowType | null;
      playbookId?: string | null;

      status?: string;
      priority?: number | null;
      dueAt?: string | null;
      ownerUser?: string | null;
    }
  ) => apiClient.patch(`/api/admin/cases/${caseRef}`, payload),

  archive: (caseRef: string) =>
    apiClient.post(`/api/admin/cases/${caseRef}/archive`),
  unarchive: (caseRef: string) =>
    apiClient.post(`/api/admin/cases/${caseRef}/unarchive`),
  addMember: (
    caseRef: string,
    payload: {
      users: Array<{ id: string }>;
      role: string;
    }
  ) => apiClient.post(`/api/admin/cases/${caseRef}/members`, payload),

  removeMember: (caseRef: string, userId: string) =>
    apiClient.delete(`/api/admin/cases/${caseRef}/members/${userId}`),
  boardMove: (
    caseRef: string,
    payload: { toStatus: string; orderedIds: string[] }
  ) => apiClient.post(`/api/admin/cases/${caseRef}/board-move`, payload),
  applyPlaybook: (
    caseRef: string,
    payload: {
      playbookRef: string; // uuid vagy key
      overwrite?: boolean;
      createChecklist?: boolean;
      overwriteChecklist?: boolean;

      createKickoffMeeting?: boolean;
      kickoffTitle?: string;
      kickoffAgendaTitle?: string;
      kickoffStartsAt?: string;
      kickoffEndsAt?: string;
    }
  ) => apiClient.post(`/api/admin/cases/${caseRef}/apply-playbook`, payload),

  getPlaybookChecklist: (caseRef: string) =>
    apiClient.get<CasePlaybookChecklist>(
      `/api/admin/cases/${caseRef}/playbook-checklist`
    ),

  patchPlaybookChecklistStep: (
    caseRef: string,
    stepRef: string,
    payload: { isDone: boolean; notes?: string | null }
  ) =>
    apiClient.patch(
      `/api/admin/cases/${caseRef}/playbook-checklist/${stepRef}`,
      payload
    ),
  overviewTree: (
    caseRef: string,
    params?: { includeArchived?: 0 | 1; includeMeetingComments?: 0 | 1 }
  ) => apiClient.get(`/api/admin/cases/${caseRef}/overview-tree`, { params }),
};
export async function applyPlaybook(caseRef: string, body: any) {
  return apiClient.put(`/api/admin/cases/${caseRef}/playbook`, body);
}

export async function getPlaybookChecklist(caseRef: string) {
  return apiClient.get(`/api/admin/cases/${caseRef}/playbook-checklist`);
}

export async function setPlaybookChecklistStep(
  caseRef: string,
  stepId: string,
  body: { isDone: boolean }
) {
  return apiClient.put(
    `/api/admin/cases/${caseRef}/playbook-checklist/${stepId}`,
    body
  );
}

export default casesApi;
