import apiClient from "@/lib/apiClient";
import attachmentsApi from "@/api/routes/attachments";

export type ActionItemStatus =
  | "open"
  | "in_progress"
  | "blocked"
  | "done"
  | "cancelled";

export type ActionItemAssigneeRole =
  | "owner"
  | "contributor"
  | "reviewer"
  | "watcher";

export type ActionItemAssignee = {
  user_id: string;
  role: ActionItemAssigneeRole;
  name: string;
  email: string | null;
  username: string | null;
};

export type ActionItem = {
  id: string;
  code?: string;

  title: string;
  description: string | null;
  status: ActionItemStatus;

  case_id: string;
  thread_id: string | null;
  decision_id: string | null;
  meeting_id: string | null;

  assigned_to: string | null;
  created_by: string | null;

  due_at: string | null;
  created_at: string;
  updated_at: string;

  archived_at?: string | null;
  archived_by?: string | null;

  case_title?: string | null;
  caseRef?: string;
};

export type ChecklistItem = { id: string; text: string; done: boolean };

export type NoteComment = {
  id: string;
  author_id: string | null;
  author_name?: string | null;
  body: string;
  is_public: boolean;
  checklist: ChecklistItem[];
  created_at: string;
  updated_at?: string;
  archived_at?: string | null;
};

export const actionItemsApi = {
  list: (params?: {
    mine?: 1 | 0;
    includeArchived?: 0 | 1;
    status?: ActionItemStatus;
    dueBefore?: string;
    caseId?: string;
    q?: string;
  }) =>
    apiClient.get<{ items: ActionItem[] }>(`/api/admin/action-items`, {
      params,
    }),

  create: (payload: {
    title: string;
    description?: string;
    status?: ActionItemStatus;
    caseId: string; // backend oldalon lehet még uuid, de listák ref-esek
    threadId?: string;
    decisionId?: string;
    meetingId?: string;
    assignedTo?: string;
    dueAt?: string;
  }) =>
    apiClient.post<{ id: string; code?: string }>(
      `/api/admin/action-items`,
      payload
    ),

  // ✅ actionItemRef = uuid OR code
  patch: (
    actionItemRef: string,
    payload: {
      title?: string;
      description?: string;
      status?: ActionItemStatus;
      assignedTo?: string | null;
      dueAt?: string | null;
    }
  ) => apiClient.patch(`/api/admin/action-items/${actionItemRef}`, payload),

  archive: (actionItemRef: string) =>
    apiClient.post(`/api/admin/action-items/${actionItemRef}/archive`),
  unarchive: (actionItemRef: string) =>
    apiClient.post(`/api/admin/action-items/${actionItemRef}/unarchive`),

  assignees: {
    get: (actionItemRef: string) =>
      apiClient.get<{ assignees: ActionItemAssignee[] }>(
        `/api/admin/action-items/${actionItemRef}/assignees`
      ),

    patch: (
      actionItemRef: string,
      payload: {
        assignees: Array<{ userId: string; role?: ActionItemAssigneeRole }>;
      }
    ) =>
      apiClient.patch(
        `/api/admin/action-items/${actionItemRef}/assignees`,
        payload
      ),
  },
  attachments: {
    upload: (actionItemRef: string, file: File) =>
      attachmentsApi.uploadToActionItem(actionItemRef, file),
  },

  comments: {
    list: (actionItemRef: string) =>
      apiClient.get<{ comments: NoteComment[] }>(
        `/api/admin/action-items/${actionItemRef}/comments`
      ),
    create: (
      actionItemRef: string,
      payload: {
        body: string;
        isPublic?: boolean;
        checklist?: ChecklistItem[];
      }
    ) =>
      apiClient.post(
        `/api/admin/action-items/${actionItemRef}/comments`,
        payload
      ),
  },
};

export default actionItemsApi;

// import apiClient from "@/lib/apiClient";

// export type ActionItemStatus =
//   | "open"
//   | "in_progress"
//   | "blocked"
//   | "done"
//   | "cancelled";

// export type ActionItem = {
//   id: string;
//   title: string;
//   description: string | null;
//   status: ActionItemStatus;
//   case_id: string;
//   thread_id: string | null;
//   decision_id: string | null;
//   meeting_id: string | null;
//   assigned_to: string | null;
//   created_by: string | null;
//   due_at: string | null;
//   created_at: string;
//   updated_at: string;
//   case_title?: string | null;
// };

// export const actionItemsApi = {
//   list: (params?: {
//     mine?: 1 | 0;
//     status?: ActionItemStatus;
//     dueBefore?: string;
//     caseId?: string;
//     q?: string;
//   }) =>
//     apiClient.get<{ items: ActionItem[] }>(`/api/admin/action-items`, {
//       params,
//     }),

//   create: (payload: {
//     title: string;
//     description?: string;
//     status?: ActionItemStatus;
//     caseId: string;
//     threadId?: string;
//     decisionId?: string;
//     meetingId?: string;
//     assignedTo?: string;
//     dueAt?: string;
//   }) => apiClient.post<{ id: string }>(`/api/admin/action-items`, payload),

//   patch: (
//     actionItemId: string,
//     payload: {
//       title?: string;
//       description?: string;
//       status?: ActionItemStatus;
//       assignedTo?: string | null;
//       dueAt?: string | null;
//     }
//   ) => apiClient.patch(`/api/admin/action-items/${actionItemId}`, payload),
// };

// export default actionItemsApi;
